package it.unicam.cs.tesei.graphs;

import java.util.*;

/**
 * On oggetto di questa classe è un grafo non diretto. La classe usa la
 * rappresentazione del grafo con liste di adiacenza per realizzare i metodi
 * dell'interfaccia <code>Graph<V,E></code>.
 * 
 * @author Luca Tesei
 *
 * @param <V>
 *            la classe di oggetti che etichettano i nodi del grafo. Ogni nodo
 *            deve avere un'etichetta diversa e non null.
 * @param <E>
 *            la classe di oggetti che etichettano gli archi del grafo. Le
 *            etichette degli archi possono essere null.
 */
public class GraphListUndirected<V, E> implements Graph<V, E> {

    /*
     * L'insieme dei nodi del grafo. Si usa una list invece che un set per poter
     * avere un indice automaticamente associato ad ogni nodo. Si controlla che
     * non vengano inseriti duplicati o elementi nulli. Si usa la classe interna
     * Node per associare al nodo informazioni aggiuntive come un indice
     * numerico univoco e crescente che può essere usato per creare degli array
     * o liste contenente i nodi o come il colore.
     */
    private ArrayList<Node> nodes;

    /*
     * Rappresentazione con liste di adiacenza, gli elementi della lista sono
     * coppie formate dall'etichetta del nodo con cui il nodo chiave è connesso
     * e l'etichetta di tipo E dell'arco corrispondente.
     */
    private HashMap<V, ArrayList<AdjacentListElement>> la;

    /*
     * Classe privata che serve per rappresentare un nodo del grafo, con
     * informazioni aggiuntive, ad esempio il colore corrente del nodo.
     */
    private class Node {
        // Puntatore all'etichetta unica del nodo nel grafo
        private final V el;

        // Colore associato al nodo
        private int color;

        private Node(V el) {
            this.el = el;
        }

        private Node(V el, int color) {
            this.el = el;
            this.color = color;
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.lang.Object#hashCode()
         */
        @Override
        public int hashCode() {
            final int prime = 31;
            int result = 1;
            result = prime * result + getOuterType().hashCode();
            result = prime * result + ((el == null) ? 0 : el.hashCode());
            return result;
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.lang.Object#equals(java.lang.Object)
         */
        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (!(obj instanceof GraphListUndirected.Node))
                return false;
            Node other = (GraphListUndirected<V, E>.Node) obj;
            if (!getOuterType().equals(other.getOuterType()))
                return false;
            if (el == null) {
                if (other.el != null)
                    return false;
            } else if (!el.equals(other.el))
                return false;
            return true;
        }

        private GraphListUndirected<V, E> getOuterType() {
            return GraphListUndirected.this;
        }

    }

    /*
     * Classe privata che serve per ottenere una coppia di oggetti che
     * rappresentano gli elementi delle liste di adiacenza: vertice del grafo ed
     * etichette dell'arco.
     */
    private class AdjacentListElement {
        private final V nodeLabel;

        private final E edgeLabel;

        private AdjacentListElement(V nodeLabel, E edgeLabel) {
            this.nodeLabel = nodeLabel;
            this.edgeLabel = edgeLabel;
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.lang.Object#hashCode()
         */
        @Override
        public int hashCode() {
            final int prime = 31;
            int result = 1;
            result = prime * result + getOuterType().hashCode();
            result = prime * result
                    + ((edgeLabel == null) ? 0 : edgeLabel.hashCode());
            result = prime * result
                    + ((nodeLabel == null) ? 0 : nodeLabel.hashCode());
            return result;
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.lang.Object#equals(java.lang.Object) Uguaglianza su
         * entrambe le componenti.
         */
        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (!(obj instanceof GraphListUndirected.AdjacentListElement))
                return false;
            AdjacentListElement other = (AdjacentListElement) obj;
            if (!getOuterType().equals(other.getOuterType()))
                return false;
            if (edgeLabel == null) {
                if (other.edgeLabel != null)
                    return false;
            } else if (!edgeLabel.equals(other.edgeLabel))
                return false;
            if (nodeLabel == null) {
                if (other.nodeLabel != null)
                    return false;
            } else if (!nodeLabel.equals(other.nodeLabel))
                return false;
            return true;
        }

        private GraphListUndirected getOuterType() {
            return GraphListUndirected.this;
        }
    }

    /**
     * Crea un grafo vuoto
     */
    public GraphListUndirected() {
        this.nodes = new ArrayList<Node>();
        this.la = new HashMap<V, ArrayList<AdjacentListElement>>();
    }

    /**
     * Crea un grafo a partire da un insieme di nodi e un insieme di archi dati.
     * 
     * @param nodes
     *            insieme di nodi
     * @param edges
     *            insieme di archi
     */
    public GraphListUndirected(Set<V> nodes, Set<Edge<V, E>> edges) {
        if (nodes == null || edges == null)
            throw new NullPointerException(
                    "Tentativo di creazione di grafo con insieme dei nodi"
                            + " o degli archi nullo");
        // Creo le strutture dati
        this.nodes = new ArrayList<Node>();
        this.la = new HashMap<V, ArrayList<AdjacentListElement>>();
        // Aggiungo tutti i nodi
        for (V n : nodes) {
            this.addNode(n);
        }
        // Aggiungo tutti gli archi
        for (Edge<V, E> e : edges) {
            this.addEdge(e.getLabel1(), e.getLabel2(), e.getLabel());
        }
    }

    @Override
    public int size() {
        return la.size();
    }

    @Override
    public boolean isEmpty() {
        return this.size() == 0;
    }

    @Override
    public boolean isDirected() {
        return false;
    }

    @Override
    public boolean addNode(V label) {
        if (label == null)
            throw new NullPointerException(
                    "Tentativo di inserimento di nodo null");
        if (this.containsNode(label)) {
            // Il nodo è già presente
            return false;
        } else {
            // Il nodo non è presente
            // Inserisco il nodo nella lista nodes
            nodes.add(new Node(label));
            // Inserisco il nodo nella mappa con lista di adiacenza vuota
            la.put(label, new ArrayList<AdjacentListElement>());
            return true;
        }
    }

    @Override
    public boolean removeNode(V label) {
        if (label == null)
            throw new NullPointerException(
                    "Tentativo di rimozione di nodo null");
        if (!this.containsNode(label))
            return false;
        /*
         * Il nodo c'è. Prima di eliminarlo dalla lista provvedo a rimuovere
         * tutti gli archi a cui lui è connesso.
         */
        // Prendo la lista di adiacenza del nodo
        List<AdjacentListElement> lan = this.la.get(label);
        // Scorro la lista
        Iterator<AdjacentListElement> vicini = lan.iterator();
        V vicinoLabel = null;
        while (vicini.hasNext()) {
            vicinoLabel = vicini.next().nodeLabel;
            /*
             * Rimuovo l'arco che c'è nella lista di adiacenza del vicino con
             * me.
             */
            // Ricavo la lista di adiacenza del vicino
            ArrayList<AdjacentListElement> laVicino = la.get(vicinoLabel);
            /*
             * Cerco nella lista di adiacenza del vicino tutti gli gli archi
             * connessi con label.
             */
            Iterator<AdjacentListElement> iViciniDelVicino = laVicino
                    .iterator();
            V vicinoDelVicinoLabel = null;
            while (iViciniDelVicino.hasNext()) {
                vicinoDelVicinoLabel = iViciniDelVicino.next().nodeLabel;
                if (vicinoDelVicinoLabel.equals(label)) {
                    /*
                     * L'etichetta è uguale a quella del nodo da rimuovere.
                     * rimuovo dalla lista di adiacenza del vicino usando il
                     * remove dell'iteratore.
                     */
                    iViciniDelVicino.remove();
                }
            }
        }
        /*
         * A questo punto il nodo non è più connesso a nessuno e quindi non
         * rimane che eliminare la sua lista di adiacenza e toglierlo dalla
         * lista dei nodi.
         */
        this.la.remove(label);
        this.nodes.remove(this.getNodeIndex(label));
        return true;
    }

    @Override
    public boolean containsNode(V label) {
        if (label == null)
            throw new NullPointerException("Tentativo di ricerca di nodo null");
        return this.nodes.contains(new Node(label));
    }

    @Override
    public int getNodeIndex(V label) {
        if (label == null)
            throw new NullPointerException("Tentativo di ricerca dell'indice"
                    + " di un nodo nullo.");
        // Cerco il nodo nella lista nodes
        int i = 0;
        boolean trovato = false;
        V nodeLabel = null;
        while (i < this.nodes.size() && !trovato) {
            nodeLabel = this.nodes.get(i).el;
            if (nodeLabel.equals(label))
                trovato = true; // Esco dal ciclo quando ho trovato il nod
            else
                i++; // altrimenti incremento il contatore per l'indice
        }
        if (trovato)
            return i;
        else
            throw new IllegalArgumentException("Tentativo di ricerca"
                    + " dell'indice di un nodo non presente");
    }

    @Override
    public V getNodeAtIndex(int i) {
        return this.nodes.get(i).el;
    }

    @Override
    public int getColor(V label) {
        // Il colore è quello del nodo con etichetta label
        if (label == null)
            throw new NullPointerException(
                    "Tentativo di rimozione di nodo null");
        // Cerco l'indice del nodo, se non lo trova lancia l'eccezione
        int i = this.getNodeIndex(label);
        // Restituisco il colore del nodo
        return this.nodes.get(i).color;
    }

    @Override
    public void setColor(V label, int color) {
        // Il colore è quello del nodo con etichetta label
        if (label == null)
            throw new NullPointerException(
                    "Tentativo di rimozione di nodo null");
        // Cerco l'indice del nodo, se non lo trova lancia l'eccezione
        int i = this.getNodeIndex(label);
        // Assegno il colore al nodo
        this.nodes.get(i).color = color;
    }

    @Override
    public int getDegree(V label) {
        if (label == null)
            throw new NullPointerException("Tentativo di calcolare il grado"
                    + " di un nodo nullo");
        if (!this.containsNode(label))
            throw new IllegalArgumentException("Tentativo di calcolare il"
                    + " grado di un nodo non presente nel grafo");
        /*
         * Il grado in questo grafo non diretto è uguale alla lunghezza della
         * lista di adiacenza.
         */
        return this.la.get(label).size();
    }

    @Override
    public Set<V> neighbors(V label) {
        if (label == null)
            throw new NullPointerException("Tentativo di calcolare i vicini"
                    + " di un nodo nullo");
        if (!this.containsNode(label))
            throw new IllegalArgumentException("Tentativo di calcolare i"
                    + " vicini di un nodo non presente nel grafo");
        /*
         * I vicini del nodo sono tutti quelli che appaiono in almeno un
         * elemento della lista di adiacenza di label.
         */
        // Creo il set risultato vuoto
        HashSet<V> neighbors = new HashSet<V>();
        // Scorro la lista di adiacenza di label e aggiungo i nodi che trovo
        List<AdjacentListElement> neighborNodes = this.la.get(label);
        for (int i = 0; i < neighborNodes.size(); i++)
            neighbors.add(neighborNodes.get(i).nodeLabel);
        return neighbors;
    }

    @Override
    public Set<V> successors(V label) {
        /*
         * Questo grafo non è diretto quindi lancio l'eccezione di operazione
         * non supportata.
         */
        throw new UnsupportedOperationException(
                "Tentativo di calcolare i successori di un nodo in un "
                        + "grafo non diretto");
    }

    @Override
    public Set<V> predecessors(V label) {
        /*
         * Questo grafo non è diretto quindi lancio l'eccezione di operazione
         * non supportata.
         */
        throw new UnsupportedOperationException(
                "Tentativo di calcolare i predecessori di un nodo in un "
                        + "grafo non diretto");
    }

    @Override
    public Set<V> getNodes() {
        // Creo l'insieme di elementi V da ritornare
        HashSet<V> ret = new HashSet<V>();
        // Per ogni elemento del set di nodi, aggiungo nel set risultato
        // l'etichetta associata, di tipo V
        for (Node n : this.nodes) {
            ret.add(n.el);
        }
        return ret;
    }

    @Override
    public boolean addEdge(V label1, V label2, E label) {
        // Controllo di etichette non null
        if (label1 == null || label2 == null)
            throw new NullPointerException("Tentativo di inserire un arco"
                    + " tra uno o entrambi nodi nulli.");
        // Controllo l'esistenza dei nodi
        if (!this.containsNode(label1) || !this.containsNode(label2))
            throw new IllegalArgumentException("Tentativo di inserire un"
                    + " arco tra uno o entrambi nodi non esistenti");
        // Controllo l'esistenza dell'arco
        if (this.containsEdge(label1, label2, label))
            return false;
        // Inserisco l'arco
        // Cerco la lista di adiacenza del nodo label1
        ArrayList<AdjacentListElement> adjacentList1 = la.get(label1);
        // Aggiungo il nodo label2
        adjacentList1.add(new AdjacentListElement(label2, label));
        // Cerco la lista di adiacenza del nodo label2
        ArrayList<AdjacentListElement> adjacentList2 = la.get(label2);
        // Aggiungo il nodo label1
        adjacentList2.add(new AdjacentListElement(label1, label));
        return true;
    }

    @Override
    public boolean removeEdge(V label1, V label2, E label) {
        // Controllo di etichette non null
        if (label1 == null || label2 == null)
            throw new NullPointerException("Tentativo di rimuovere un arco"
                    + " tra uno o entrambi nodi nulli.");
        // Controllo l'esistenza dei nodi
        if (!this.containsNode(label1) || !this.containsNode(label2))
            throw new IllegalArgumentException("Tentativo di rimuovere un"
                    + " arco tra uno o entrambi nodi non esistenti");
        // Controllo l'esistenza dell'arco
        if (!this.containsEdge(label1, label2, label))
            return false;
        // Elimino l'arco
        // Cerco la lista di adiacenza del nodo label1
        ArrayList<AdjacentListElement> adjacentList1 = la.get(label1);
        // Rimuovo il nodo con etichette label2, label
        adjacentList1.remove(new AdjacentListElement(label2, label));
        // Cerco la lista di adiacenza del nodo label2
        ArrayList<AdjacentListElement> adjacentList2 = la.get(label2);
        // Rimuovo il nodo con etichette label1, label
        adjacentList2.remove(new AdjacentListElement(label1, label));
        return true;
    }

    @Override
    public boolean containsEdge(V label1, V label2, E label) {
        // Controllo di etichette non null
        if (label1 == null || label2 == null)
            throw new NullPointerException("Tentativo di trovare un arco"
                    + " tra uno o entrambi nodi nulli.");
        // Controllo l'esistenza dei nodi
        if (!this.containsNode(label1) || !this.containsNode(label2))
            throw new IllegalArgumentException("Tentativo di trovare un"
                    + " arco tra uno o entrambi nodi non esistenti");
        Set<Edge<V, E>> edges = this.getEdges(label1, label2);
        return edges.contains(new Edge<V, E>(label1, label2, label, false));
    }

    @Override
    public Set<Edge<V, E>> getEdges(V label1, V label2) {
        // Controllo di etichette non null
        if (label1 == null || label2 == null)
            throw new NullPointerException("Tentativo di trovare un arco"
                    + " tra uno o entrambi nodi nulli.");
        // Controllo l'esistenza dei nodi
        if (!this.containsNode(label1) || !this.containsNode(label2))
            throw new IllegalArgumentException("Tentativo di trovare un"
                    + " arco tra uno o entrambi nodi non esistenti");
        // Creo l'insieme risultato
        HashSet<Edge<V, E>> archi = new HashSet<Edge<V, E>>();
        // Prendo la lista di adiacenza di uno dei due nodi, scelta arbitraria
        ArrayList<AdjacentListElement> lan = this.la.get(label1);
        // Cerco tutti gli archi verso label2 e li inserisco nell'insieme
        AdjacentListElement vicino = null;
        for (int i = 0; i < lan.size(); i++) {
            vicino = lan.get(i);
            if (vicino.nodeLabel.equals(label2))
                archi.add(new Edge<V, E>(label1, label2, vicino.edgeLabel,
                        false));
        }
        // Restituisco il risultato
        return archi;
    }

    @Override
    public Set<Edge<V, E>> getEdges(V label) {
        // Controllo di etichetta non null
        if (label == null)
            throw new NullPointerException("Tentativo di calcolare gli archi"
                    + " da un nodo nullo.");
        // Controllo l'esistenza del nodo
        if (!this.containsNode(label))
            throw new IllegalArgumentException("Tentativo di calcolare gli"
                    + " archi da un nodo non esistente.");
        // Creo l'insieme risultato
        HashSet<Edge<V, E>> archi = new HashSet<Edge<V, E>>();
        // Prendo la lista di adiacenza del nodo
        ArrayList<AdjacentListElement> lan = this.la.get(label);
        // Aggiungo tutti gli archi nell'insieme
        AdjacentListElement vicino = null;
        for (int i = 0; i < lan.size(); i++) {
            vicino = lan.get(i);
            archi.add(new Edge<V, E>(label, vicino.nodeLabel, vicino.edgeLabel,
                    false));
        }
        // Restituisco il risultato
        return archi;
    }

    @Override
    public Set<Edge<V, E>> outgoingEdges(V label) {
        throw new UnsupportedOperationException("Tentativo di calcolare"
                + " gli archi uscenti da un nodo in un grafo non diretto");
    }

    @Override
    public Set<Edge<V, E>> ingoingEdges(V label) {
        throw new UnsupportedOperationException("Tentativo di calcolare"
                + " gli archi entranti in un nodo in un grafo non diretto");
    }

    @Override
    public Set<Edge<V, E>> getEdges() {
        // Creo l'insieme risultato
        HashSet<Edge<V, E>> archi = new HashSet<Edge<V, E>>();
        // Scorro tutte le liste di adiacenza
        Set<V> las = la.keySet();
        Iterator<V> it = las.iterator();
        V n = null;
        while (it.hasNext()) {
            n = it.next();
            // Prendo la lista di adiacenza del nodo
            ArrayList<AdjacentListElement> lan = this.la.get(n);
            // Aggiungo tutti gli archi nell'insieme
            AdjacentListElement vicino = null;
            for (int i = 0; i < lan.size(); i++) {
                vicino = lan.get(i);
                // Controllo se l'arco è già presente per evitare i doppioni
                if (!this.containsEdge(n, vicino.nodeLabel, vicino.edgeLabel))
                    archi.add(new Edge<V, E>(n, vicino.nodeLabel,
                            vicino.edgeLabel, false));
            }
        }
        // Restituisco il risultato
        return archi;
    }

    @Override
    public int edgeCount() {
        /*
         * Conto tutti gli elmenti nelle liste di adiacenza e divido per due,
         * infatti tutti gli archi sono ripetuti due volte essendo il grafo non
         * orientato.
         */
        int count = 0;
        for (Node n : nodes)
            count += la.get(n.el).size();
        return count / 2;
    }

    @Override
    public void clear() {
        this.nodes.clear();
        this.la.clear();
    }
}
