package it.unicam.cs.tesei.trees;

import java.util.ArrayList;
import java.util.List;

public class BinSearchTree<E extends Comparable<E>> {
    // Generico elemento che etichetta la radice di questo albero
    private E el;

    // Sottoalbero sinistro, può essere nullo
    private BinSearchTree<E> left;

    // Sottoalbero destro, può essere nullo
    private BinSearchTree<E> right;

    /**
     * Costruisce un albero binario di ricerca con solo la radice
     * 
     * @param el
     *            elemento da associare alla radice
     */
    public BinSearchTree(E el) {
        this.left = null;
        this.el = el;
        this.right = null;
    }

    /**
     * Costruisce un albero binario di ricerca che ha un solo sottoalbero, quello destro.
     * 
     * @param el
     *            elemento da associare alla radice di questo albero binario
     * @param right
     *            sottoalbero destro della radice
     */
    public BinSearchTree(E el, BinSearchTree<E> right) {
        // Controllo la proprieta dei Binary Search Trees
        if (el.compareTo(right.getMinimum()) < 0) {
            this.left = null;
            this.el = el;
            this.right = right;
        } else
            throw new IllegalArgumentException(
                    "Elementi non corretti per la costruzione di un albero binario di ricerca");

    }

    /**
     * Costruisce un albero binario di ricerca che ha un solo sottoalbero, quello sinistro.
     * 
     * @param left
     *            sottoalbero sinistro della radice
     * @param el
     *            elemento da associare alla radice di questo albero binario
     */
    public BinSearchTree(BinSearchTree<E> left, E el) {
        if (el.compareTo(left.getMaximum()) >= 0) {
            this.left = left;
            this.el = el;
            this.right = null;
        } else
            throw new IllegalArgumentException(
                    "Elementi non corretti per la costruzione di un albero binario di ricerca");

    }

    /**
     * Costruisce un albero binario di ricerca che ha due sottoalberi.
     * 
     * @param left
     *            sottoalbero sinistro della radice
     * @param el
     *            elemento da associare alla radice di questo albero binario
     * @param right
     *            sottoalbero destro della radice
     */
    public BinSearchTree(BinSearchTree<E> left, E el, BinSearchTree<E> right) {
        if ((el.compareTo(right.getMinimum()) < 0)
                && (el.compareTo(left.getMaximum()) >= 0)) {
            this.left = left;
            this.el = el;
            this.right = right;
        } else
            throw new IllegalArgumentException(
                    "Elementi non corretti per la costruzione di un albero binario di ricerca");

    }

    /**
     * Restituisce l'elemento minimo di questo albero.
     * 
     * @return l'elemento più piccolo presente in questo albero utilizzando
     *         l'ordinamento naturale della classe E.
     */
    public E getMinimum() {
        // Se non ho sottoalbero sinistro allora il minimo è la radice di questo
        // albero
        if (this.left == null)
            return this.el;
        else
            // Altrimenti il minimo è il minimo del sottoalbero sinistro
            return this.left.getMinimum();
    }

    /**
     * Restituisce l'elemento masssimo di questo albero.
     * 
     * @return l'elemento più grande presente in questo albero utilizzando
     *         l'ordinamento naturale della classe E.
     */
    public E getMaximum() {
        // Se non ho sottoalbero sinistro allora il massimo è la radice di
        // questo albero
        if (this.right == null)
            return this.el;
        else
            return this.right.getMaximum();
    }

    /**
     * Cerca un elemento in questo albero.
     * 
     * @param o
     *            l'elemento da cercare
     * @return true se l'elemento è presente, false altrimenti. La presenza
     *         viene testata utilizzando il metodo <code>equals()</code> della
     *         classe E che si suppone sia stato ridefinito in maniera
     *         opportuna.
     */
    public boolean search(E o) {
        int k = this.el.compareTo(o);
        if (k == 0)
            // Presente
            return true;
        // Altrimenti
        if (k < 0) {
            // o si può trovare solo nel sottoalbero destro
            if (this.right == null)
                return false;
            else
                return this.right.search(o);
        } else {
            // o si può trovare solo nel sottoalbero sinistro
            if (this.left == null)
                return false;
            else
                return this.left.search(o);
        }
    }

    /**
     * Restituisce la lista ordinata degli elementi in questo albero.
     * 
     * @return una lista ordinata degli elementi dell'albero.
     */
    public List<E> getOrderedElements() {
        List<E> l = new ArrayList<E>();
        recGetOrderedElements(l);
        return l;
    }

    // Implementazione ricorsiva del metodo getOrderedElements()
    private void recGetOrderedElements(List<E> l) {
        if (this.left == null && this.right == null) {
            // Caso Foglia
            l.add(this.el);
            return;
        }
        if (this.left != null && this.right == null) {
            // Solo sottoalbero sinistro
            this.left.recGetOrderedElements(l);
            l.add(this.el);
            return;
        }
        if (this.left == null && this.right != null) {
            // Solo sottoalbero destro
            l.add(this.el);
            this.right.recGetOrderedElements(l);
            return;
        }
        // Entrambi i sottoalberi
        this.left.recGetOrderedElements(l);
        l.add(this.el);
        this.right.recGetOrderedElements(l);
        return;
    }
}
