package it.unicam.cs.tesei.equations;

/**
 * Equazione di secondo grado non modificabile con uguaglianza, ordinamento
 * naturale e pretty print tramite toString
 * 
 * @author luca
 *
 */
public class EquazioneDiSecondoGradoComparable implements
        Comparable<EquazioneDiSecondoGradoComparable> {

    private final double a;

    private final double b;

    private final double c;

    private final double delta;

    /**
     * Costruisce una equazione con i tre parametri
     * 
     * @param a
     *            il parametro a deve essere diverso da zero
     */
    public EquazioneDiSecondoGradoComparable(double a, double b, double c) {
        if (a != 0)
            this.a = a;
        else
            throw new IllegalArgumentException(
                    "Parametro a uguale a zero, l'equazione non è di secondo grado");
        this.b = b;
        this.c = c;
        this.delta = b * b - 4 * a * c;
    }

    /**
     * Restituisce il parametro a
     */
    public double getA() {
        return a;
    }

    /**
     * Restituisce il parametro b
     */
    public double getB() {
        return b;
    }

    /**
     * Restituisce il parametro c
     */
    public double getC() {
        return c;
    }

    /**
     * Restituisce la prima soluzione. Nel caso di nessuna soluzione restituisce
     * la costante Double.NaN
     */
    public double getFirstSolution() {
        if (delta < 0)
            return Double.NaN;
        else if (delta == 0)
            return -b / (2 * a);
        else
            // caso delta > 0
            return (-b + Math.sqrt(delta)) / (2 * a);
    }

    /**
     * Restituisce la seconda soluzione. Nel caso di nessuna soluzione
     * restituisce la costante Double.NaN
     */
    public double getSecondSolution() {
        if (delta < 0)
            return Double.NaN;
        else if (delta == 0)
            return -b / (2 * a);
        else
            // caso delta > 0
            return (-b - Math.sqrt(delta)) / (2 * a);
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return a + "x^2 + " + b + "x + " + c;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        long temp;
        temp = Double.doubleToLongBits(a);
        result = prime * result + (int) (temp ^ (temp >>> 32));
        temp = Double.doubleToLongBits(b);
        result = prime * result + (int) (temp ^ (temp >>> 32));
        temp = Double.doubleToLongBits(c);
        result = prime * result + (int) (temp ^ (temp >>> 32));
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj) {
        // Uguaglianza in base ai tre parametri
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (!(obj instanceof EquazioneDiSecondoGradoComparable))
            return false;
        EquazioneDiSecondoGradoComparable other = 
                (EquazioneDiSecondoGradoComparable) obj;
        if (Double.doubleToLongBits(a) != Double.doubleToLongBits(other.a))
            return false;
        if (Double.doubleToLongBits(b) != Double.doubleToLongBits(other.b))
            return false;
        if (Double.doubleToLongBits(c) != Double.doubleToLongBits(other.c))
            return false;
        return true;
    }

    /**
     * Ordinamento naturale tra equazioni, in base al parametro a. Nel caso le
     * due equazioni abbiano parametro a uguale, si guarda il parametro b. Se
     * anche b è uguale allora si guarda il parametro c. Se anche quest'ultimo è
     * uguale allora è la stessa equazione.
     */
    @Override
    public int compareTo(EquazioneDiSecondoGradoComparable o) {
        if (this.getA() != o.getA())
            return Double.compare(this.getA(), o.getA());
        else if (this.getB() != o.getB())
            return Double.compare(this.getB(), o.getB());
        else if (this.getC() != o.getC())
            return Double.compare(this.getC(), o.getC());
        else
            return 0;
    }

}
