package it.unicam.cs.tesei.graphs;

import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

/**
 * Algoritmo generico che calcolo un albero minimo di copertura per un grafo non
 * orientato e pesato. Il peso degli archi viene determinato dal fatto che la
 * classe E deve implementare l'interfaccia Weights.
 * 
 * @author luca
 *
 * @param <V>
 * @param <E>
 */
public class GenericMinimumSpanningTree<V, E extends Weights> {
    public Comparator<Edge<V, E>> comp = new Comparator<Edge<V, E>>() {
        public int compare(Edge<V, E> e1, Edge<V, E> e2) {
            Double we1 = new Double(e1.getLabel().getWeight());
            return we1.compareTo(new Double(e2.getLabel().getWeight()));
        }
    };

    /**
     * Calcola un minimum spanning tree per il grafo passato.
     * 
     * @param g
     *            grafo pesato i cui nodi sono tutti bianchi.
     * @return un insieme di archi del grafo che rappresentano un minimum
     *         spanning tree.
     */
    public Set<Edge<V, E>> findMinimumSpanningTree(Graph<V, E> g, V source) {
        if (g.isDirected())
            throw new IllegalArgumentException(
                    "L'algoritmo di MST funziona solo con grafi non diretti.");
        Set<Edge<V, E>> mst = new HashSet<Edge<V, E>>();
        SortedSet<Edge<V, E>> taglio = new TreeSet<Edge<V, E>>(this.comp);
        Set<V> nodes = g.getNodes(); // Ci viene fornito un insieme tale che se
                                     // togliamo i nodi essi non vengono tolti
                                     // dal grafo
        V currentNode = source;
        nodes.remove(currentNode);
        while (!nodes.isEmpty()) {
            Set<Edge<V, E>> archiDalCurrentNode = g.getEdges(currentNode);
            // Tolgo tutti i nodi che vanno a finire in un nodo, diverso da
            // currentNode, che non appartiene a nodes, cioè è un nodo già
            // coperto
            Iterator<Edge<V, E>> i = archiDalCurrentNode.iterator();
            Edge<V, E> arco = null;
            while (i.hasNext()) {
                arco = i.next();
                V n1 = arco.getLabel1();
                V n2 = arco.getLabel2();
                if ((n1.equals(currentNode) && !nodes.contains(n2))
                        || (n2.equals(currentNode) && !nodes.contains(n1))) {
                    // l'arco corrente non va considerato perché raggiunge un
                    // nodo già coperto
                    i.remove();
                }
            }
            // Dentro archiDalCurrentNode sono rimasti solo gli archi
            // candidati ad entrare nel taglio
            // Li aggiungo nel taglio
            taglio.addAll(archiDalCurrentNode);
            // Determiniamo l'arco leggero
            Edge<V, E> arcoLeggero = taglio.first();
            // Lo tolgo dal taglio
            taglio.remove(arcoLeggero);
            // Lo aggiungo al risultato
            mst.add(arcoLeggero);
            // Aggiorna currentNode con il nuovo nodo scelto
            if (nodes.contains(arcoLeggero.getLabel1()))
                // Il nodo nuovo da coprire è quello in posizione 1
                currentNode = arcoLeggero.getLabel1();
            else
                // Il nodo nuovo da coprire è quello in posizione 2
                currentNode = arcoLeggero.getLabel2();
            // Rimuovi currentNode da nodes
            nodes.remove(currentNode);
            // Toglo dal taglio tutti gli archi che contengono il nuovo
            // currentNode
            Iterator<Edge<V, E>> j = taglio.iterator();
            while (j.hasNext()) {
                arco = j.next();
                if (arco.getLabel1().equals(currentNode)
                        || arco.getLabel2().equals(currentNode))
                    // tolgo l'arco dal taglio
                    j.remove();
            }
        }
        return mst;
    }
}
