package it.unicam.cs.tesei.sorting;

import java.util.ArrayList;
import java.util.List;

/**
 * Heap binomiale. E' ammesso avere lo heap vuoto.
 * 
 * @author Luca Tesei
 *
 * @param <E>
 */
public class Heap<E extends Comparable<E>> {

    // Lista contentente gli elementi dello heap
    private List<E> heap;

    // Numero di confronti effettuati durante la vita di questo heap
    // Serve per la valutazione del numero di confronti per valutare l'algoritmo
    // di ordinamento HeapSort basato su heap
    private int countCompare;

    /**
     * Costruisce uno heap a partire dalla sequenza di elementi data.
     * 
     * @param list
     *            sequenza di elementi che in generale non è uno heap
     * 
     */
    public Heap(List<E> list) {
        this.countCompare = 0;
        this.heap = list;
        int i;
        for (i = (list.size() / 2) - 1; i >= 0; i--) {
            this.heapify(i);
        }
    }

    /**
     * Costruisce uno heap contentente un solo elemento
     * 
     * @param e
     *            elemento iniziale dello heap
     */
    public Heap(E e) {
        this.countCompare = 0;
        this.heap = new ArrayList<E>();
        this.heap.add(e);
    }

    /**
     * Costruisce lo heap vuoto
     */
    public Heap() {
        this.countCompare = 0;
        this.heap = new ArrayList<E>();
    }

    /**
     * Restutisce il numero di elementi presenti in questo heap.
     * 
     * @return il numero di elementi in questo heap
     */
    public int size() {
        return this.heap.size();
    }

    /**
     * Restituisce l'indice del figlio sinistro del nodo all'indice dato in uno
     * heap.
     * 
     * @param p
     *            l'indice del nodo padre di cui si cerca l'indice del figlio
     *            sinistro.
     * @return l'indice del figlio sinistro del nodo p in uno heap
     */
    public int left(int p) {
        return (2 * (p + 1)) - 1;
    }

    /**
     * Restituisce l'indice del figlio destro del nodo all'indice dato in uno
     * heap.
     * 
     * @param p
     *            l'indice del nodo padre di cui si cerca l'indice del figlio
     *            destro.
     * @return l'indice del figlio destro del nodo p in uno heap
     */
    public int right(int p) {
        return 2 * (p + 1);
    }

    /**
     * Restituisce l'indice del nodo padre di un nodo in una posizione data in
     * uno heap.
     * 
     * @param p
     *            l'indice di cui si vuole conoscere l'indice del nodo padre
     * @return l'indice del nodo padre del nodo all'indice p in uno heap
     */
    public int padre(int p) {
        return ((p + 1) / 2) - 1;
    }

    /**
     * Inserisce un nuovo elemento in questo heap.
     * 
     * @param o
     *            il nuovo elemento da inserire
     */
    public void insert(E o) {
        // Inserisco il nuovo elemento in fondo alla sequenza
        heap.add(o);
        // Parto dall'indice dell'elemento e confronto con i padri fino a quando
        // non trovo un padre maggiore o uguale oppure fino a quando non divento
        // la radice
        int p = heap.size() - 1;
        while (p != 0 && heap.get(p).compareTo(heap.get(padre(p))) > 0) {
            // Incremento il numero di confronti
            this.countCompare++;
            // Scambio gli elementi p e padre(p)
            E temp = heap.get(p);
            heap.set(p, heap.get(padre(p)));
            heap.set(padre(p), temp);
            p = padre(p);
        }
    }

    /**
     * Ricostituisce questo heap a partire da un nodo p assumendo che i suoi
     * figli sinistro e destro (se esistono) sono a loro volta heap.
     * 
     * @param p
     *            l'indice del nodo da cui ricostituire lo heap
     * 
     * @throws IndexOutOfBoundsException
     *             se l'indice passato non è valido in questo heap
     */
    public void heapify(int p) {
        // Lancio un eccezione se l'indice non esiste
        if (p < 0 || p >= heap.size()) {
            throw new IndexOutOfBoundsException(
                    "Chiamata di heapify con indice non valido = " + p);
        }
        // Caso base: p è un nodo foglia
        if ((right(p) >= heap.size()) && (left(p) >= heap.size()))
            return;
        // Casi ricorsivi
        // Esiste solo il figlio sinistro
        if ((right(p) >= heap.size()) && (left(p) < heap.size())) {
            // Se l'elemento in posizione p è maggiore o uguale all'elemento
            // nella
            // radice del figlio sinistro, non devo fare niente
            if (heap.get(p).compareTo(heap.get(left(p))) >= 0) {
                // Incremento il numero di confronti
                this.countCompare++;
                return;
            }
            // Altrimenti scambio i due elementi e mi richiamo ricorsivamente
            // sul figlio sinistro
            E temp = heap.get(p);
            heap.set(p, heap.get(left(p)));
            heap.set(left(p), temp);
            this.heapify(left(p));
            return;
        }
        // Caso in cui esistono entrambi i figli, calcolo il massimo tra p e i
        // suoi due figli
        int max = p;
        // Incremento il numero di confronti
        this.countCompare++;
        if (heap.get(left(p)).compareTo(heap.get(max)) > 0)
            max = left(p);
        // Incremento il numero di confronti
        this.countCompare++;
        if (heap.get(right(p)).compareTo(heap.get(max)) > 0)
            max = right(p);
        // Caso facile
        if (max == p)
            return;
        if (max == left(p)) {
            // Scambio i due elementi e mi richiamo ricorsivamente
            // sul figlio sinistro
            E temp = heap.get(p);
            heap.set(p, heap.get(left(p)));
            heap.set(left(p), temp);
            this.heapify(left(p));
            return;
        } else {
            // Scambio i due elementi e mi richiamo ricorsivamente
            // sul figlio destro
            E temp = heap.get(p);
            heap.set(p, heap.get(right(p)));
            heap.set(right(p), temp);
            this.heapify(right(p));
            return;
        }
    }

    /**
     * Estrae l'elemento massimo da questo heap e ricostituisce l'heap con gli
     * elementi rimanenti.
     * 
     * @return l'elemento massimo di questo heap
     */
    public E extractMax() {
        if (this.size() == 0)
            throw new NullPointerException(
                    "Tentativo di estrarre il massimo da uno heap vuoto");
        E ret = heap.get(0);
        // Metto l'ultimo elemento in testa alla lista e poi ricostituisco
        // l'heap, se non è diventato vuoto
        this.heap.set(0, this.heap.get(this.heap.size() - 1));
        this.heap.remove(this.heap.size() - 1);
        if (this.size() > 0)
            this.heapify(0);
        return ret;
    }

    /**
     * Restituisce il numero di confronti fatti finora da questo heap. Usato per
     * la valutazione della complessità dell'algoritmo HeapSort.
     * 
     * @return il numero di confronti fra elmenti della classe E fatti finora da
     *         questo heap.
     */
    public int getNCompare() {
        return this.countCompare;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        // Visita anticipata
        StringBuffer sb = new StringBuffer();
        if (this.size() > 0)
            this.visit(sb, 0);
        return sb.toString();
    }

    private void visit(StringBuffer sb, int p) {
        // Lancio un eccezione se l'indice non esiste
        if (p < 0 || p >= heap.size()) {
            throw new IndexOutOfBoundsException(
                    "Chiamata di visit con indice non valido = " + p);
        }
        sb.append(this.heap.get(p).toString() + "\n");
        if (left(p) < heap.size())
            // Esiste il figlio sinistro
            visit(sb, left(p));
        if (right(p) < heap.size())
            // Esiste il figlio destro
            this.visit(sb, right(p));
    }

}
