package it.unicam.cs.tesei.trees;

public class BinTreeMeasurable<E extends IntMeasurable> {
    // Generico elemento che etichetta la radice di questo albero
    private E el;

    // Sottoalbero sinistro, può essere nullo
    private BinTreeMeasurable<E> left;

    // Sottoalbero destro, può essere nullo
    private BinTreeMeasurable<E> right;

    /**
     * Costruisce un albero binario con solo la radice
     * 
     * @param el
     *            elemento da associare alla radice
     */
    public BinTreeMeasurable(E el) {
        this.left = null;
        this.el = el;
        this.right = null;
    }

    /**
     * Costruisce un albero binario che ha un solo sottoalbero, quello destro.
     * 
     * @param el
     *            elemento da associare alla radice di questo albero binario
     * @param right
     *            sottoalbero destro della radice
     */
    public BinTreeMeasurable(E el, BinTreeMeasurable<E> right) {
        this.left = null;
        this.el = el;
        this.right = right;
    }

    /**
     * Costruisce un albero binario che ha un solo sottoalbero, quello sinistro.
     * 
     * @param left
     *            sottoalbero sinistro della radice
     * @param el
     *            elemento da associare alla radice di questo albero binario
     */
    public BinTreeMeasurable(BinTreeMeasurable<E> left, E el) {
        this.left = left;
        this.el = el;
        this.right = null;

    }

    /**
     * Costruisce un albero binario che ha due sottoalberi.
     * 
     * @param left
     *            sottoalbero sinistro della radice
     * @param el
     *            elemento da associare alla radice di questo albero binario
     * @param right
     *            sottoalbero destro della radice
     */
    public BinTreeMeasurable(BinTreeMeasurable<E> left, E el,
            BinTreeMeasurable<E> right) {
        this.left = left;
        this.el = el;
        this.right = right;
    }

    /**
     * @return the element associated to the root of this binary tree
     */
    public E getEl() {
        return el;
    }

    /**
     * @return the left subtree
     */
    public BinTreeMeasurable<E> getLeft() {
        return left;
    }

    /**
     * @return the right subtree
     */
    public BinTreeMeasurable<E> getRight() {
        return right;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((el == null) ? 0 : el.hashCode());
        result = prime * result + ((left == null) ? 0 : left.hashCode());
        result = prime * result + ((right == null) ? 0 : right.hashCode());
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (!(obj instanceof BinTreeMeasurable))
            return false;
        BinTreeMeasurable other = (BinTreeMeasurable) obj;
        if (el == null) {
            if (other.el != null)
                return false;
        } else if (!el.equals(other.el))
            return false;
        if (left == null) {
            if (other.left != null)
                return false;
        } else if (!left.equals(other.left))
            return false;
        if (right == null) {
            if (other.right != null)
                return false;
        } else if (!right.equals(other.right))
            return false;
        return true;
    }

    /**
     * Restituisce la rappresentazione di questo albero come stringa con
     * parentesi tonde
     * 
     * @return una stringa di parentesi tonde bilanciate rappresentante questo
     *         albero
     */
    public String getParentesizzazione() {
        StringBuffer sb = new StringBuffer();
        this.getParentesizzazione(sb);
        return sb.toString();
    }

    // Implementazione ricorsiva di una visita di questo albero che genera la
    // descrizione di questo albero con parentesi tonde bilanciate
    private void getParentesizzazione(StringBuffer sb) {
        sb.append(" ( ");
        if (this.left != null)
            this.left.getParentesizzazione(sb);
        if (this.right != null)
            this.right.getParentesizzazione(sb);
        sb.append(" ) ");
    }

    /**
     * Restituisce la rappresentazione di questo albero come stringa con
     * parentesi tonde ed elementi
     * 
     * @return una stringa di parentesi tonde bilanciate ed elementi che
     *         rappresenta questo albero
     */
    public String getParentesizzazioneConElementi() {
        StringBuffer sb = new StringBuffer();
        this.getParentesizzazioneConElementi(sb);
        return sb.toString();
    }

    // Implementazione ricorsiva di una visita di questo albero che genera la
    // descrizione di questo albero con le parentesi tonde bilanciate e gli
    // elementi
    private void getParentesizzazioneConElementi(StringBuffer sb) {
        sb.append(" ( ");
        Object o = this.el;
        sb.append(o.toString());
        if (this.left != null)
            this.left.getParentesizzazioneConElementi(sb);
        if (this.right != null)
            this.right.getParentesizzazioneConElementi(sb);
        sb.append(" ) ");
    }

    /**
     * Esegue la visita anticipata di questo albero stampando su una stringa, in
     * sequenza, i nodi visitati.
     */
    public String visitaAnticipata() {
        StringBuffer sb = new StringBuffer();
        this.visitaAnticipata(sb);
        return sb.toString();
    }

    // Implementazione ricorsiva della visita anticipata che scrive la sequenza
    // degli elementi in uno StringBuffer dato
    private void visitaAnticipata(StringBuffer sb) {
        // Cast su Object per chiamare toString()
        Object o = this.el;
        sb.append(" " + o.toString() + " ");
        if (this.left != null)
            this.left.visitaAnticipata(sb);
        if (this.right != null)
            this.right.visitaAnticipata(sb);
    }

    /**
     * Esegue la visita posticipata di questo albero stampando in sequenza i
     * nodi visitati.
     */
    public String visitaPosticipata() {
        StringBuffer sb = new StringBuffer();
        this.visitaPosticipata(sb);
        return sb.toString();
    }

    // Implementazione ricorsiva della visita posticipata che scrive la sequenza
    // degli elementi in uno StringBuffer dato
    private void visitaPosticipata(StringBuffer sb) {
        if (this.left != null)
            this.left.visitaPosticipata(sb);
        if (this.right != null)
            this.right.visitaPosticipata(sb);
        // Cast su Object per chiamare toString()
        Object o = this.el;
        sb.append(" " + o.toString() + " ");
    }

    /**
     * Calcola l'altezza di quest'albero. Un albero con solo la radice ha
     * altezza zero. L'altezza di un albero è la massima profondità (ogni arco
     * aumenta la profondità di uno) di una foglia rispetto alla radice.
     * 
     * @return l'altezza di questo albero
     */
    public int getHeight() {
        // Caso base: albero con solo la radice
        if (this.left == null && this.right == null)
            return 0;
        // Caso ricorsivo con due sottoalberi
        if (this.left != null && this.right != null)
            return Math.max(this.left.getHeight(), this.right.getHeight()) + 1;
        // Caso ricorsivo con solo il sottoalbero sinistro
        if (this.left != null && this.right == null)
            return this.left.getHeight() + 1;
        else
            // Caso ricorsivo con solo il sottolbero destro
            return this.right.getHeight() + 1;
    }

    /**
     * Restituisce la somma delle misure dei nodi interni dell'albero, cioè
     * quelli che non sono foglie.
     * 
     * @return la somma delle misure dei nodi interni dell'albero
     * 
     */
    public int getSumOfInternalNodes() {
        if (this.left == null && this.right == null)
            return 0;
        if (this.left != null && this.right == null)
            return this.el.getMeasure() + this.left.getSumOfInternalNodes();
        if (this.left == null && this.right != null)
            return this.el.getMeasure() + this.right.getSumOfInternalNodes();
        else
            // Ultimo caso possible omesso: (this.left != null && this.right !=
            // null)
            return this.el.getMeasure() + this.left.getSumOfInternalNodes()
                    + this.right.getSumOfInternalNodes();
    }

    /**
     * Determina se questo albero è una foglia.
     * 
     * @return true se questo albero è una foglia, false altrimenti.
     */
    public boolean isLeaf() {
        if (this.left == null && this.right == null)
            return true;
        else
            return false;
    }

    /**
     * Un albero binario è bilanciato se il sottoalbero destro e sinistro sono
     * bilanciati. Al livello delle foglie potrebbe mancare qualche foglia.
     * 
     * @return true se questo albero è bilanciato, false altrimenti.
     */
    public boolean isBalanced() {
        if (this.left == null && this.right == null)
            return true;
        if (this.left != null && this.right == null)
            return this.left.isLeaf();
        if (this.left == null && this.right != null)
            return this.right.isLeaf();
        else
            // Ultimo caso possible omesso: (this.left != null && this.right !=
            // null)
            return this.left.isBalanced() && this.right.isBalanced();
    }

    /**
     * Calcola il massimo costo associato ai cammini su questo albero. Il costo
     * associato a un cammino di un albero è pari alla somma delle misure dei
     * nodi del cammino. Un cammino è una sequenza di nodi che parte dalla
     * radice e termina in una foglia seguendo la relazione padre-figlio
     * dell'albero.
     * 
     * @return il massimo costo associato ai cammini di questo albero. 
     */
    public int maxPathCost() {
        // TODO per casa
        return 0;
    }

}
