/**
 * 
 */
package it.unicam.cs.asdl1617.binarytrees;

import java.util.List;

/**
 * Classe che rappresenta alberi binari di ricerca. Gli elementi da inserire
 * sono generici di una classe in cui è definito l'ordinamento naturale. Questa
 * implementazione non ammette che vengano inseriti elementi duplicati o nulli.
 * 
 * @author luca
 *
 */
public class BinarySearchTree<E extends Comparable<E>> {
    /*
     * Elemento associato al nodo radice
     */
    private E el;

    /*
     * Sottoalbero sinistro
     */
    private BinarySearchTree<E> left;

    /*
     * Sottoalbero destro
     */
    private BinarySearchTree<E> right;

    /**
     * Costruisce un BST che contiene solo la radice/foglia.
     * 
     * @param anEl
     *            elemento da associare alla radice.
     */
    public BinarySearchTree(E anEl) {
        if (anEl == null)
            throw new NullPointerException(
                    "Tentativo di inserire un elemento null nel BST");
        this.el = anEl;
        this.left = null;
        this.right = null;
    }

    /**
     * Costruisce un BST a partire da un nodo radice e due sottoalberi BST.
     * 
     * @param anEl
     *            elemento da associare al nodo radice
     * @param aLeft
     *            sottoalbero sinistro
     * @param aRight
     *            sottoalbero destro
     */
    public BinarySearchTree(E anEl, BinarySearchTree<E> aLeft,
            BinarySearchTree<E> aRight) {
        if (anEl == null)
            throw new NullPointerException(
                    "Tentativo di inserire un elemento null nel BST");
        this.el = anEl;
        this.left = aLeft;
        this.right = aRight;

    }

    /**
     * Restituisce l'altezza dell'albero.
     * 
     * @return la lunghezza del massimo cammino dalla radice a una foglia.
     */
    public int getHeight() {
        if (this.left == null && this.right == null)
            // Sono una radice foglia
            return 0;
        else if (this.left == null)
            // Ho solo il figlio destro
            return 1 + this.right.getHeight();
        else if (this.right == null)
            // Ho solo il figlio sinistro
            return 1 + this.left.getHeight();
        else
            // Ho tutti e due i figli
            return 1 + Math.max(this.left.getHeight(), this.right.getHeight());
    }

    /**
     * Restituisce l'elemento più piccolo presente nella struttura.
     * 
     * @return l'elemento più a sinistra che non ha un sottoalbero sinistro.
     */
    public E getMin() {
        if (this.left == null)
            // Sono il nodo più a sinistra che non ha il figlio sinistro
            return this.el;
        else
            // Mi richiamo sul sottoalbero sinistro
            return this.left.getMin();
    }

    /**
     * Restituisce l'elemento più grande presente nella struttura.
     * 
     * @return l'elemento più a destra che non ha un sottoalbero destro.
     */
    public E getMax() {
        if (this.right == null)
            // Sono il nodo più a destra che non ha il figlio destro
            return this.el;
        else
            // Mi richiamo sul sottoalbero destro
            return this.right.getMax();
    }

    /**
     * Aggiunge ad una lista data la lista degli elementi del BST nell'ordine
     * naturale. Per far questo esegue una visita simmetrica del BST.
     * 
     * @param l
     *            una lista (possibilmente vuota) su cui inserire gli elementi
     *            in ordine.
     */
    public void getOrderedElements(List<E> l) {
        // Visita simmetrica
        // Visito il sottoalbero sinistro, se esiste
        if (this.left != null)
            this.left.getOrderedElements(l);
        // Adesso visito il nodo corrente
        l.add(this.el);
        // Visito il sottoalbero destro, se esiste
        if (this.right != null)
            this.right.getOrderedElements(l);
    }

    /**
     * Aggiunge un elemento al BST.
     * 
     * @param el
     *            elemento da inserire
     * @return true se l'elemento è stato effettivamente inserito, false se era
     *         già presente.
     */
    public boolean add(E el) {
        // In un albero BST, un nuovo nodo inserito non presente è sempre
        // inserito come foglia
        if (el == null)
            throw new NullPointerException(
                    "Tentativo di inserire un elemento null nel BST");
        // Confronto l'elemento con la radice
        int x = this.el.compareTo(el);
        // Caso di uguaglianza
        if (x == 0)
            // L'elemento è già presente in radice
            return false;
        // Caso non di uguaglianza
        if (x < 0)
            // L'elemento da inserire andrebbe nel sottoalbero destro
            if (this.right == null) {
                // Inseriamo l'elemento come sottoalbero destro
                this.right = new BinarySearchTree<E>(el);
                return true;
            } else
                return this.right.add(el);
        else
        // L'elemento da inserire andrebbe nel sottoalbero sinistro
        if (this.left == null) {
            // Inseriamo l'elemento come sottoalbero sinistro
            this.left = new BinarySearchTree<E>(el);
            return true;
        } else
            return this.left.add(el);
    }
}
