/**
 * 
 */
package it.unicam.cs.asdl1617.hashing;

import java.util.ArrayList;
import java.util.LinkedList;

/**
 * Classe che realizza una tabella hash con gestione delle collisioni tramite
 * liste concatenate (buckets).
 * 
 * @author luca
 *
 */
public class MyHash<E> {
    // Usiamo ArrayList invece che un array normale
    // per poter utilizzare agevolmente il generics <E>
    // Ogni elemento della tabella è un bucket
    private ArrayList<LinkedList<E>> table;

    // Dimensione della tabella hash
    private final int m;

    /**
     * Crea una tabella hash di dimensione data.
     * 
     * @param m
     *            un intero positivo, possibilmente un numero primo vicino a una
     *            potenza di 2.
     */
    public MyHash(int m) {
        if (m <= 0)
            throw new IllegalArgumentException(
                    "Tentativo di creare una hashtable vuota o"
                            + " con dimensione negativa");
        this.m = m;
        this.table = new ArrayList<LinkedList<E>>();
        for (int i = 0; i < m; i++)
            this.table.add(new LinkedList<E>());
    }

    /**
     * Inserisce un elemento nella tabella hash.
     * 
     * @param el
     *            l'elemento da inserire, non può essere null.
     */
    public void add(E el) {
        if (el == null)
            throw new NullPointerException("Tentativo di inserire un elemento"
                    + " null nella tabella hash");
        // Calcola l'hash
        int h = Math.abs(el.hashCode() % this.m);
        // Inserire l'elemento el in posizione h, ma gestendo
        // le possibili collisioni con una lista concatenata
        // Cerco se l'elemento è nel bucket
        if (!this.table.get(h).contains(el))
            // Inserisco l'elemento nel bucket
            this.table.get(h).add(el);
        // else non faccio niente
    }

    /**
     * Cerca se un elemento è nella tabella hash.
     * 
     * @param el
     * @return true se l'elemento è attualmente nella tabella, false altrimenti.
     */
    public boolean contains(E el) {
        if (el == null)
            return false;
        // Calcola l'hash
        int h = Math.abs(el.hashCode() % this.m);
        return this.table.get(h).contains(el);
    }

    /**
     * Elimina un elemento dalla tabella hash.
     * 
     * @param el
     * @return true se l'elemento era presente ed è stato tolto, false
     *         altrimenti.
     */
    public boolean remove(E el) {
        if (el == null)
            return false;
        // Calcola l'hash
        int h = Math.abs(el.hashCode() % this.m);
        return this.table.get(h).remove(el);
    }
    
    /**
     * Restituisce la dimensione di questa tabella hash.
     * 
     * @return
     */
    public int getDimension() {
        return this.m;
    }

}
