/**
 * 
 */
package it.unicam.cs.asdl1617.hashing;

import java.util.ArrayList;

/**
 * Realizza una tabella hash con gestione delle collisioni tramite
 * indirizzamento aperto.
 * 
 * La funzione di hash primaria h(k) viene data come parametro iniziale. La
 * funzione di indirizzamento aperto è h'(k,i) = (h(k) + i) mod m.
 * 
 * @author luca
 *
 */
public class MyHashOpen<E> {
    // Tabella
    private ArrayList<E> table;

    // Dimensione della tabella
    private final int m;

    // Funzione di hash primaria h(k)
    private final PrimaryHashFunction h;

    /**
     * Crea l'hashtable della dimensione specificata.
     * 
     * @param m
     *            numero positivo, possibilmente un numero primo vicino a una
     *            potenza di due.
     * @param h
     *            funzione hash primaria
     */
    public MyHashOpen(int m, PrimaryHashFunction h) {
        if (m <= 0)
            throw new IllegalArgumentException(
                    "Tentativo di creare una hashtable vuota o con dimensione negativa");
        this.m = m;
        this.h = h;
        this.table = new ArrayList<E>();
        for (int i = 0; i < m; i++)
            this.table.add(null);
    }

    /**
     * Inserisce un elemento nella tabella hash.
     * 
     * @param el
     *            l'elemento da inserire
     * @throws HashOverflowException
     *             se si tenta di inserire un elemento in una tabella in cui non
     *             c'è più posto libero.
     */
    public void add(E el) {
        if (el == null)
            throw new NullPointerException(
                    "Tentativo di inserire null nella hashtable");
        int tentativo = 0;
        int hashPrimaria = h.hash(this.m, el);
        boolean finito = false;
        while (tentativo < this.m && !finito) {
            int posizione = (hashPrimaria + tentativo) % m;
            if (this.table.get(posizione) == null) {
                // c'è posto, inserisco
                this.table.set(posizione, el);
                finito = true;
            } else // posto occupato
            if (this.table.get(posizione).equals(el))
                // l'elemento è già presente, non faccio niente
                finito = true;
            else
                // devo andare avanti
                tentativo++;
        }
        if (!finito)
            // overflow
            throw new HashOverflowException(
                    "Raggiunto il limite di elementi che si possono inserire in questa hashtable");
    }

    /**
     * Cerca se un elemento è in una tabella hash.
     * 
     * @param el
     * @return
     */
    public boolean contains(E el) {
        if (el == null)
            return false;
        int tentativo = 0;
        int hashPrimaria = h.hash(this.m, el);
        while (tentativo < m) {
            int posizione = (hashPrimaria + tentativo) % m;
            if (this.table.get(posizione) == null)
                // l'elemento non c'è
                return false;
            // il posto è occupato
            if (this.table.get(posizione).equals(el))
                // l'elemento c'è
                return true;
            // Il posto è occupato, ma non da el, quindi vado avanti nella
            // ricerca
            tentativo++;
        }
        // L'elemento non c'è
        return false;
    }

    /**
     * Restituisce la dimensione della tabella
     * 
     * @return
     */
    public int getDimension() {
        return this.m;
    }

    /**
     * Restituisce la funzione hash primaria usata da questa tabella hash.
     * 
     * @return
     */
    public PrimaryHashFunction getPrimaryHashFunction() {
        return this.h;
    }
}
