/**
 * 
 */
package it.unicam.cs.asdl1819.graphs;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * Classe che implementa un grafo non orientato con liste di adiacenza.
 * 
 * Non implementa le operazioni di remove e le operazioni che dipendono da
 * indici.
 * 
 * @author Luca Tesei
 *
 */
public class AdjacentListNonDirectedGraph<V, E> implements Graph<V, E> {

    private Map<GraphNode<V>, Set<GraphEdge<V, E>>> adjList;

    /**
     * Crea un grafo vuoto.
     * 
     */
    public AdjacentListNonDirectedGraph() {

        this.adjList = new HashMap<GraphNode<V>, Set<GraphEdge<V, E>>>();
    }

    @Override
    public int nodeCount() {
        return this.adjList.keySet().size();
    }

    @Override
    public int edgeCount() {
        // contiamo tutti gli archi nelle liste di adiacenza e dividiamo per due
        // perché in un grafo non orientato le liste di adiacenza contengono lo
        // stesso arco due volte
        int result = 0;
        for (GraphNode<V> n : this.adjList.keySet())
            result = result + this.adjList.get(n).size();
        // il numero calcolato deve essere pari
        assert result % 2 == 0;
        return result / 2;
    }

    @Override
    public int size() {
        return this.nodeCount() + this.edgeCount();
    }

    @Override
    public boolean isEmpty() {
        return this.adjList.isEmpty();
    }

    @Override
    public void clear() {
        this.adjList.clear();
    }

    @Override
    public boolean isDirected() {
        // questo grafo non è orientato
        return false;
    }

    @Override
    public Set<GraphNode<V>> getNodes() {
        return this.adjList.keySet();
    }

    @Override
    public boolean addNode(GraphNode<V> node) {
        if (node == null)
            throw new NullPointerException(
                    "Tentativo di inserire un nodo null");
        if (this.adjList.keySet().contains(node))
            // il nodo è già presente
            return false;
        // aggiungo il nodo con insieme degli archi associati vuoto
        this.adjList.put(node, new HashSet<GraphEdge<V, E>>());
        return true;
    }

    @Override
    public boolean removeNode(GraphNode<V> node) {
        throw new UnsupportedOperationException(
                "Operazioni di remove non implementate");
    }

    @Override
    public boolean containsNode(GraphNode<V> node) {
        if (node == null)
            throw new NullPointerException("Tentativo di cercare un nodo null");
        return this.adjList.keySet().contains(node);
    }

    @Override
    public GraphNode<V> getNode(V label) {
        if (label == null)
            throw new NullPointerException("Tentativo di cercare un nodo null");
        // scorro l'insieme dei nodi e cerco quello con etichetta equals a label
        for (GraphNode<V> n : this.adjList.keySet())
            if (n.getLabel().equals(label))
                return n;
        // il grafo non contiene nessun nodo con l'etichetta label
        throw new IllegalArgumentException(
                "Tentativo di cercare un nodo che non esiste");
    }

    @Override
    public int getNodeIndex(V label) {
        throw new UnsupportedOperationException(
                "Operazioni con indici non implementate");
    }

    @Override
    public GraphNode<V> getNodeAtIndex(int i) {
        throw new UnsupportedOperationException(
                "Operazioni con indici non implementate");
    }

    @Override
    public Set<GraphEdge<V, E>> getEdgesBetween(int index1, int index2) {
        throw new UnsupportedOperationException(
                "Operazioni con indici non implementate");
    }

    @Override
    public Set<GraphEdge<V, E>> getEdgesBetween(GraphNode<V> node1,
            GraphNode<V> node2) {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public Set<GraphNode<V>> getAdjacentNodes(GraphNode<V> node) {
        if (node == null)
            throw new NullPointerException(
                    "Tentativo di cercare i nodi adiacenti a un nodo null");
        Set<GraphEdge<V, E>> archiCollegati = this.adjList.get(node);
        if (archiCollegati == null)
            throw new IllegalArgumentException(
                    "Tentativo di ottenere i nodi vicini di un nodo che non esiste");
        Set<GraphNode<V>> result = new HashSet<GraphNode<V>>();
        for (GraphEdge<V, E> e : archiCollegati)
            // nel grafo non orientato il nodo node può essere il primo o il
            // secondo nodo dell'arco
            if (e.getNode1().equals(node))
                result.add(e.getNode2());
            else
                result.add(e.getNode1());
        return result;
    }

    @Override
    public Set<GraphNode<V>> getPredecessorNodes(GraphNode<V> node) {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public Set<GraphEdge<V, E>> getEdges() {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public boolean addEdge(GraphEdge<V, E> edge) {
        if (edge == null)
            throw new NullPointerException(
                    "Tentativo di inserire un arco null");
        // Controllo che entrambi i nodi esistono
        if (!this.containsNode(edge.getNode1())
                || !!this.containsNode(edge.getNode2()))
            throw new IllegalArgumentException(
                    "Inserimento di un arco che collega almeno un nodo non esistente");
        if (edge.isDirected())
            throw new IllegalArgumentException(
                    "Inserimento di un arco orientato in un grafo non orientato");
        // Inserisco l'arco nella lista di adiacenza sia del nodo1 che del nodo2
        boolean result = this.adjList.get(edge.getNode1()).add(edge);
        result = result && this.adjList.get(edge.getNode2()).add(edge);
        return result;
    }

    @Override
    public boolean removeEdge(GraphEdge<V, E> edge) {
        // TODO Auto-generated method stub
        return false;
    }

    @Override
    public boolean containsEdge(GraphEdge<V, E> edge) {
        // TODO Auto-generated method stub
        return false;
    }

    @Override
    public Set<GraphEdge<V, E>> getEdges(GraphNode<V> node) {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public Set<GraphEdge<V, E>> getIngoingEdges(GraphNode<V> node) {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public int getDegree(GraphNode<V> node) {
        // TODO Auto-generated method stub
        return 0;
    }

}
