/**
 * 
 */
package it.unicam.cs.asdl1819.adtlists;

/**
 * Abstract Data Type List, ovvero liste immutabili costruite a partire dalla
 * lista vuota con inserimento in testa (operazione cons). L'implementazione di
 * default è data da due classi: EmptyADTList che implemente la lista vuota e
 * ConsList che implementa l'operazione cons con campi immutabili. In questa
 * interface sono definiti vari metodi ricorsivi generici sulle ADTList.
 * 
 * @author Luca Tesei
 *
 */
public interface ADTList<E> {

    /**
     * Lista vuota.
     */
    @SuppressWarnings("rawtypes")
    public static final ADTList EMPTY_LIST = new EmptyADTList();

    /**
     * Determina se questa lista è vuota.
     * 
     * @return true se questa lista è vuota.
     */
    public boolean isEmpty();

    /**
     * Restituisce l'elemento in testa alla lista.
     * 
     * @return l'elemento in testa
     * 
     * @throws IllegalStateException
     *                                   se il metodo viene chiamato su una
     *                                   lista vuota.
     */
    public E head();

    /**
     * Restituisce la coda di questa lista, cioè questa lista in cui è stato
     * tolto l'elemento di testa.
     * 
     * @return la coda di questa lista
     * @throws IllegalStateException
     *                                   se il metodo viene chiamato su una
     *                                   lista vuota.
     */
    public ADTList<E> tail();

    /**
     * Aggiunge un elemento in testa a questa lista
     * 
     * @param head
     *                 l'elemento da aggiungere in testa
     * @return una lista con l'elemento {@code head} in testa e questa lista
     *         come coda
     */
    public ADTList<E> cons(E head);

    /**
     * Restituisce una stringa contenente il toString di tutti gli elementi di
     * questa lista separati da uno spazio.
     * 
     * @return una stringa con tutti gli elementi di questa lista
     */
    default String print() {
        // caso base
        if (this.isEmpty())
            return "";
        // caso ricorsivo
        return this.head().toString() + " " + this.tail().print();
    }

    /**
     * Cerca se un elemento è in questa lista.
     * 
     * @param element
     *                    l'elemento da cercare
     * @return true se questa lista contiene {@code element}
     */
    default boolean find(E e) {
        // caso base
        if (this.isEmpty())
            return false;
        // caso ricorsivo
        if (this.head().equals(e))
            return true;
        else
            return this.tail().find(e);
    }


    /**
     * Cancella la prima occorrenza di un elemento in questa lista, se presente.
     * 
     * @param element
     *                    l'elemento da cancellare
     * @return una lista uguale a questa in cui la prima occorrenza di
     *         {@code element}, se presente, è stata cancellata.
     */
    default ADTList<E> remove(E element) {
        // caso base
        if (this.isEmpty())
            return this;
        // caso ricorsivo
        if (this.head().equals(element))
            return this.tail();
        else
            return this.tail().remove(element).cons(this.head());
    }

    /**
     * Cancella tutti gli elementi uguali a un elemento dato in questa lista.
     * 
     * @param element
     *                    l'elemento da cancellare
     * @return una lista uguale a questa in cui tutte le occorrenze di
     *         {@code element} sono state cancellate.
     */
    default ADTList<E> removeAll(E element) {
        // caso base
        if (this.isEmpty())
            return this;
        // caso ricorsivo
        if (this.head().equals(element))
            return this.tail().removeAll(element);
        else
            return this.tail().removeAll(element).cons(this.head());
    }

    /**
     * Attacca una lista data in fondo a questa lista.
     * 
     * @param list
     *                 la lista da attaccare in fondo
     * @return una lista che contiene gli elementi di questa lista seguita dagli
     *         elementi di {@code list}.
     */
    default ADTList<E> append(ADTList<E> list) {
        // caso base
        if (this.isEmpty())
            return list;
        // caso ricorsivo
        return this.tail().append(list).cons(this.head());
    }

}
