package it.unicam.cs.asdl1819.tree;

import java.util.ArrayList;
import java.util.List;

/**
 * Classe per rappresentare alberi ordinati ed etichettati in cui ogni nodo può
 * avere un numero arbitrario di figli.
 * 
 * @author Luca Tesei
 *
 * @param <E>
 *            il tipo delle etichette dei nodi
 */
public class Tree<E> {
    private E label;

    private List<Tree<E>> children;

    /**
     * Crea un albero con solo la radice, che è anche una foglia.
     * 
     * @param label
     *                  l'informazione associata alla radice
     */
    public Tree(E label) {
        this.label = label;
        this.children = null;
    }

    /**
     * Crea un albero a partire dalla radice e dai suoi sottoalberi.
     * 
     * @param label
     *                     l'informazione associata alla radice
     * @param children
     *                     la lista dei sottoalberi della radice
     */
    public Tree(E label, List<Tree<E>> children) {
        this.label = label;
        this.children = children;
    }

    /**
     * Restituisce la lista dei sottoalberi della radice di questo albero.
     * 
     * @return la lista dei sottoalberi della radice di questo albero.
     */
    public List<Tree<E>> getChildren() {
        return this.children;
    }

    /**
     * Imposta la lista dei sottoalberi della radice di questo albero.
     * 
     * @param children
     *                     una lista di alberi che diventeranno i figli della
     *                     radice di questo albero.
     */
    public void setChildren(List<Tree<E>> children) {
        this.children = children;
    }

    /**
     * Restituisce l'altezza di questo albero, che è la massima profondità
     * raggiungibile facendo una ricerca in profondità.
     * 
     * @return l'altezza di questo albero
     */
    public int getHeight() {
        // caso base
        if (this.children == null)
            return 0;
        // caso ricorsivo
        // calcola la massima altezza dei sottoalberi
        int maxSubtreesHeight = -1;
        for (Tree<E> t : this.children) {
            int height = t.getHeight();
            if (height > maxSubtreesHeight)
                maxSubtreesHeight = height;
        }
        // restituisce l'altezza massima dei sottoalberi + 1
        return maxSubtreesHeight + 1;
    }

    /**
     * Effettua la visita in preordine di questo albero.
     * 
     * @return la lista delle etichette dei nodi visitati durante la visita in
     *         preordine.
     */
    public List<E> preOrderVisitList() {
        List<E> result = new ArrayList<E>();
        result.add(this.label);
        // caso base
        if (this.children == null)
            return result;
        // caso ricorsivo
        for (Tree<E> t : this.children)
            result.addAll(t.preOrderVisitList());
        return result;
    }

    /**
     * Effettua la visita in preordine di questo albero.
     * 
     * @return una stringa contenente la stampa delle etichette di tutti i nodi
     *         visitati secondo la visita in preordine.
     */
    public String preOrderVisit() {
        String result = this.label.toString();
        // caso base
        if (this.children == null)
            return result;
        // caso ricorsivo
        for (Tree<E> t : this.children)
            result = result + " " + t.preOrderVisit();
        return result;
    }

    /**
     * Effettua la visita in postordine di questo albero.
     * 
     * @return una stringa contenente la stampa delle etichette di tutti i nodi
     *         visitati secondo la visita in postordine.
     */
    public String postOrderVisit() {
        // caso base
        if (this.children == null)
            return this.label.toString();
        // caso ricorsivo
        String result = "";
        for (Tree<E> t : this.children)
            result = result + t.postOrderVisit() + " ";
        result = result + this.label.toString();
        return result;
    }

    /**
     * Effettua la visita in postordine di questo albero.
     * 
     * @return la lista delle etichette dei nodi visitati durante la visita in
     *         postordine.
     */
    public List<E> postOrderVisitList() {
        List<E> result = new ArrayList<E>();
        // caso base
        if (this.children == null) {
            result.add(this.label); 
            return result;
        }
        // caso ricorsivo
        for (Tree<E> t : this.children)
            result.addAll(t.postOrderVisitList());
        result.add(this.label);
        return result;
    }    
    
    /**
     * Cerca una certa label in questo albero.
     * 
     * @param label
     *                  la label da cercare
     * @return true se almeno un nodo dell'albero contiene {@code label}
     */
    public boolean find(E label) {
        if (this.label.equals(label))
            return true;
        // la radice non contiene l'elemento cercato
        if (this.children == null)
            return false;
        // cerco nei figli
        for (Tree<E> t : this.children)
            if (t.find(label))
                return true;
        // in nessun sottoalbero è presente label
        return false;
    }
}
