package it.unicam.cs.asdl1819.progdin;

import java.util.List;

/**
 * Risolve il problema della parentesizzazione ottima del prodotto di una
 * sequenza di matrici tramite la tecnica di programmazione dinamica.
 * 
 * @author Luca Tesei
 *
 */
public class MatrixMultiplicationOrderSolver {

    private final List<Integer> p;

    private int[][] m;

    private int[][] s;

    private boolean computed;

    /**
     * Costruisce un solver dando i numeri delle dimensioni delle matrici. Ad
     * esempio A_0 x A_1 x A_2 corrisponde alla lista: righe(A_0), colonne(A_0)
     * = righe(A_1), colonne(A_1) = righe(A_2), colonne(A_2). Se le matrici sono
     * n, il numero di valori da passare in p è n + 1.
     * 
     * @param p
     *              lista del numero di righe e colonne delle matrici da
     *              moltiplicare
     */
    public MatrixMultiplicationOrderSolver(List<Integer> p) {
        this.p = p;
        this.computed = false;
        // creiamo le matrici
        this.m = new int[this.p.size() - 1][this.p.size() - 1];
        this.s = new int[this.p.size() - 1][this.p.size() - 1];
    }

    /**
     * Restituisce la lista lista del numero di righe e colonne delle matrici da
     * moltiplicare.
     * 
     * @return lista del numero di righe e colonne delle matrici da moltiplicare
     */
    public List<Integer> getP() {
        return p;
    }

    /**
     * Determina se questo solver ha svolto le sue operazioni.
     * 
     * @return true se le operazioni sono state svolte, false altrimenti
     */
    public boolean isComputed() {
        return this.computed;
    }

    /**
     * Svolge le operazioni di risoluzione del problema.
     */
    public void solve() {
        // Numero di matrici = n
        int n = this.p.size() - 1;
        // Inizializzazione: riempio la diagonale principale
        for (int i = 0; i < n; i++) {
            m[i][i] = 0;
            s[i][i] = i;
        }
        // Ciclo principale controllato da h = 1 .. n - 1
        for (int h = 1; h < n; h++) {
            // Riempiamo la diagonale [i][j=i+h]
            for (int i = 0; i < n - h; i++) {
                int j = i + h;
                // Dobbiamo calcolare il k corrispondente al minimo della
                // ricorrenza
                // inizializzo la casella con +infinito
                m[i][j] = Integer.MAX_VALUE;
                // All'inzio k non si sa
                s[i][j] = -1;
                // Ciclo per calcolare il minimo
                for (int k = i; k < j; k++) {
                    int v = m[i][k] + m[k + 1][j]
                            + p.get(i) * p.get(k + 1) * p.get(j + 1);
                    if (v < m[i][j]) {
                        // Aggiorno il minimo e segno il k
                        m[i][j] = v;
                        s[i][j] = k;
                    }
                }
            }
        }
        // Aggiorno lo stato
        this.computed = true;
    }

    /**
     * Restituisce il numero minimo di moltiplicazioni da effettuare per
     * moltiplicare la sequenza di matrici date.
     * 
     * @return il numero minimo di moltiplicazioni da effettuare per
     *         moltiplicare la sequenza di matrici date
     * @throws IllegalStateException
     *                                   se il minimo non è ancora stato
     *                                   calcolato
     */
    public int getMinimumNumberOfMultiplications() {
        if (!this.computed)
            throw new IllegalStateException(
                    "Richiesta di una soluzione non ancora calcolata");
        return m[0][this.p.size() - 2];
    }

    /**
     * Restituisce una stringa che indica una parentesizzazione ottima, cioè il
     * modo di moltiplicare la sequenza di matrici date per effettuare il numero
     * minimo di moltiplicazioni.
     * 
     * @return una stringa che indica una parentesizzazione ottima della
     *         sequenza di matrici data
     */
    public String getOptimalParenthesization() {
        StringBuffer sb = new StringBuffer();
        printOptimalParens(sb, 0, this.p.size() - 2);
        return sb.toString();
    }

    /*
     * Metodo ricorsivo di stampa della parentesizzazione
     */
    private void printOptimalParens(StringBuffer sb, int i, int j) {
        if (i == j)
            sb.append(" A_" + i + " ");
        else {
            sb.append("(");
            printOptimalParens(sb, i, this.s[i][j]);
            printOptimalParens(sb, this.s[i][j] + 1, j);
            sb.append(")");
        }
    }
}
