package it.unicam.cs.asdl1819.hash;

import java.util.ArrayList;

/**
 * Realizza una tabella hash con indirizzamento primario e liste di collisione.
 * La tabella non si auto incrementa se cresce il fattore di caricamento.
 * 
 * @author Luca Tesei
 *
 * @param <E>
 *            il tipo degli elementi che possono essere inseriti nella tabella
 */
public class CollisionListHashTable<E> {
    private PrimaryHashFunction phf;

    private ArrayList<?>[] tab;

    private int numeroDiElementi;

    /**
     * Crea una tabella hash con una certa capacità e associata a una certa
     * funzione di hash primaria.
     * 
     * @param h
     *                     la funzione di hash primaria che verrà usata in
     *                     questa tabella
     * @param capacity
     *                     il massimo numero di elementi che può contenere la
     *                     tabella
     * 
     * @throws NullPointerException
     *                                      se la funzione passata è null
     * @throws IllegalArgumentException
     *                                      se la capacità è minore o uguale a
     *                                      zero
     */
    public CollisionListHashTable(PrimaryHashFunction phf, int capacity) {
        if (phf == null)
            throw new NullPointerException(
                    "Creazione di una tabella hash con funzione di hash primaria nulla");
        if (capacity <= 0)
            throw new IllegalArgumentException(
                    "Creazione di una tabella hash vuota o con capacità negativa");
        this.phf = phf;
        this.numeroDiElementi = 0;
        this.tab = new ArrayList<?>[capacity];
    }

    /**
     * Restituisce il numero attuale di elementi presenti nella tabella.
     * 
     * @return il numero attuale di elementi presenti nella tabella
     */
    public int size() {
        return this.numeroDiElementi;
    }

    /**
     * Determina se questa tabella è vuota.
     * 
     * @return true se questa tabella è vuota, false altrimenti.
     */
    public boolean isEmpty() {
        return this.numeroDiElementi == 0;
    }

    /**
     * Cancella tutti gli elementi della tabella e la rende vuota.
     */
    public void clear() {
        for (int i = 0; i < tab.length; i++)
            tab[i] = null;
    }

    /**
     * Determina se un elemento è presente nella tabella hash.
     * 
     * @param el
     *               l'elemento da cercare
     * @return true se c'è un elemento equals a el nella tabella, false
     *         altrimenti
     * @throws NullPointerException
     *                                  se l'elemento passato è nullo
     */
    public boolean contains(E el) {
        if (el == null)
            throw new NullPointerException("Ricerca di elemento null");
        int key = el.hashCode();
        int pos = this.phf.hash(key, tab.length);
        // L'elemento dovrebbe trovarsi in posizione pos
        if (tab[pos] == null)
            // l'elemento non è presente
            return false;
        // Nella posizione c'è una lista di collisione
        @SuppressWarnings("unchecked")
        ArrayList<E> collisionList = (ArrayList<E>) tab[pos];
        // Scorro la lista e cerco l'elemento
        for (E e : collisionList)
            if (e.equals(el))
                return true;
        // L'elemento non è presente
        return false;
    }

    /**
     * Inserisce un elemento in questa tabella hash.
     * 
     * @param el
     *               l'elemento da inserire
     * @return true se l'elemento è stato inserito, false se era già presente
     * 
     */
    @SuppressWarnings("unchecked")
    public boolean add(E el) {
        if (el == null)
            throw new NullPointerException("Inserimento di elemento null");
        int key = el.hashCode();
        int pos = this.phf.hash(key, tab.length);
        // L'elemento dovrebbe trovarsi in posizione pos
        if (tab[pos] == null) {
            // l'elemento non è presente, lo inseriamo
            tab[pos] = new ArrayList<E>();
            // Richiesto cast esplicito per determinare il tipo generico ?
            ((ArrayList<E>) tab[pos]).add(el);
            this.numeroDiElementi++;
            return true;
        }
        // L'elemento potrebbe trovarsi nella lista in tab[pos]
        // Nella posizione c'è una lista di collisione
        ArrayList<E> collisionList = (ArrayList<E>) tab[pos];
        // Scorro la lista e cerco l'elemento
        for (E e : collisionList)
            if (e.equals(el))
                // l'elemento è già presente
                return false;
        // L'elemento non è presente nella collisionList, quindi lo aggiungo
        collisionList.add(el);
        this.numeroDiElementi++;
        return true;
    }

    /**
     * Rimuove un elemento da questa tabella.
     * 
     * @param el
     *               l'elemento da rimuovere
     * @return true se l'elemento è stato rimosso, false se non era presente
     * 
     */
    public boolean remove(E el) {
        if (el == null)
            throw new NullPointerException("Cancellazione di elemento null");
        int key = el.hashCode();
        int pos = this.phf.hash(key, tab.length);
        // L'elemento dovrebbe trovarsi in posizione pos
        if (tab[pos] == null)
            // l'elemento non è presente
            return false;
        // L'elemento potrebbe trovarsi nella lista in tab[pos]
        // Nella posizione c'è una lista di collisione
        @SuppressWarnings("unchecked")
        ArrayList<E> collisionList = (ArrayList<E>) tab[pos];
        // Scorro la lista e cerco l'elemento
        for (int i = 0; i < collisionList.size(); i++)
            if (collisionList.get(i).equals(el)) {
                // l'elemento è presente, lo devo togliere
                collisionList.remove(i);
                this.numeroDiElementi--;
                return true;
            }
        // L'elemento non è presente nella collisionList
        return false;
    }
}
