package it.unicam.cs.asdl1819.hash;

/**
 * Realizza una tabella hash con indirizzamento aperto. La tabella viene creata
 * di una certa dimensione e non possono essere inseriti più elementi di questa
 * dimensione.
 * 
 * @author Luca Tesei
 *
 * @param <E>
 *            il tipo degli elementi che possono essere inseriti nella tabella
 */
public class OpenAddressingHashTable<E> {
    private PrimaryHashFunction phf;

    private OpenAddressingHashFunction oahf;

    private OpenAddressingHashTable<?>.Element tab[];

    private int numeroDiElementi;

    /**
     * Crea una tabella hash ad indirizzamento aperto con una capacità data Crea
     * una tabella hash con una certa capacità e associata a una certa funzione
     * di hash primaria.
     * 
     * @param phf
     *                     la funzione di hash primaria che verrà usata dalla
     *                     funzione hash con indirizzamento aperto
     * @param oahf
     *                     la funzione di hasch con indirizzamento aperto che
     *                     verrà usata in questa tabella
     * @param capacity
     *                     il massimo numero di elementi che può contenere la
     *                     tabella
     * 
     * @throws NullPointerException
     *                                      se almeno una delle due funzioni
     *                                      passate è null
     * @throws IllegalArgumentException
     *                                      se la capacità è minore o uguale a
     *                                      zero
     */
    public OpenAddressingHashTable(PrimaryHashFunction phf,
            OpenAddressingHashFunction oahf, int capacity) {
        if (phf == null || oahf == null)
            throw new NullPointerException(
                    "Creazione di una tabella hash con funzione di hash "
                            + "primaria o ad indirizzamento aperto nulla");
        if (capacity <= 0)
            throw new IllegalArgumentException(
                    "Creazione di una tabella hash vuota o con capacità "
                            + "negativa");
        this.phf = phf;
        this.oahf = oahf;
        this.tab = new OpenAddressingHashTable<?>.Element[capacity];
        this.numeroDiElementi = 0;
    }

    /**
     * Restituisce il numero attuale di elementi presenti nella tabella.
     * 
     * @return il numero attuale di elementi presenti nella tabella
     */
    public int size() {
        return this.numeroDiElementi;

    }

    /**
     * Determina se questa tabella è vuota.
     * 
     * @return true se questa tabella è vuota, false altrimenti.
     */
    public boolean isEmpty() {
        return this.numeroDiElementi == 0;

    }

    /**
     * Restituisce la capacità della tabella, cioè il numero massimo di elementi
     * inseribili.
     * 
     * @return la capacità massima della tabella
     */
    public int getCapacity() {
        return this.tab.length;
    }

    /**
     * Cancella tutti gli elementi della tabella e la rende vuota.
     */
    public void clear() {
        for (int i = 0; i < tab.length; i++)
            tab[i] = null;
    }

    /**
     * Determina se un elemento è presente nella tabella hash.
     * 
     * @param el
     *               l'elemento da cercare
     * @return true se c'è un elemento equals a el nella tabella, false
     *         altrimenti
     * @throws NullPointerException
     *                                  se l'elemento passato è nullo
     */
    public boolean contains(E el) {
        if (el == null)
            throw new NullPointerException("Ricerca di elemento null");
        int i = 0;
        int pos = -1;
        int key = el.hashCode();
        do {
            // Calcolo la posizione al tentativo i
            pos = this.oahf.openHash(this.phf, key, this.tab.length, i);
            if (this.tab[pos] == null)
                // l'elemento non c'è
                return false;
            // C'è qualcuno, controllo se è l'elemento el
            if (tab[pos].el.equals(el)) {
                // controllo se l'elemento è cancellato
                if (!tab[pos].isCancelled)
                    return true; // ok
                // altrimenti vado avanti a cercare
            }
            // Non sono uguali oppure l'elemento era cancellato, vado avanti
            // nella catena di tentativi
            i++;
        } while (i < this.tab.length);
        // non ho trovato l'elemento nella catena di tentativi
        return false;
    }

    /**
     * Inserisce un elemento in questa tabella hash.
     * 
     * @param el
     *               l'elemento da inserire
     * @return true se l'elemento è stato inserito, false se era già presente
     * @throws IllegalStateException
     *                                   se non c'è più spazio per inserire
     *                                   elementi
     */
    public boolean add(E el) {
        if (el == null)
            throw new NullPointerException("Aggiunta di elemento null");
        int i = 0;
        int pos = -1;
        int key = el.hashCode();
        do {
            // Calcolo la posizione al tentativo i
            pos = this.oahf.openHash(this.phf, key, this.tab.length, i);
            if (this.tab[pos] == null) {
                // l'elemento non c'è, lo inserisco
                this.tab[pos] = new Element(el);
                this.numeroDiElementi++;
                return true;
            }
            // C'è qualcuno, controllo se è l'elemento el
            if (tab[pos].el.equals(el)) {
                if (tab[pos].isCancelled)
                    // lo rimetto
                    tab[pos].isCancelled = false;
                else // non è cancellato quindi l'elemento è realmente presente
                    return false;
            }

            // Non sono uguali, controllo se l'elemento è cancellato
            if (tab[pos].isCancelled) {
                // posso inserire l'elemento qui
                this.tab[pos] = new Element(el);
                this.numeroDiElementi++;
                return true;
            }
            // vado avanti nella catena di tentativi
            i++;
        } while (i < this.tab.length);
        // non ho trovato l'elemento nella catena di tentativi
        throw new IllegalStateException("Tabella piena");
    }

    /**
     * Rimuove un elemento da questa tabella.
     * 
     * @param el
     *               l'elemento da rimuovere
     * @return true se l'elemento è stato rimosso, false se non era presente
     * 
     */
    public boolean remove(E el) {
        if (el == null)
            throw new NullPointerException("Rimozione di elemento null");
        int i = 0;
        int pos = -1;
        int key = el.hashCode();
        do {
            // Calcolo la posizione al tentativo i
            pos = this.oahf.openHash(this.phf, key, this.tab.length, i);
            if (this.tab[pos] == null) {
                // l'elemento non c'è
                return false;
            }
            // C'è qualcuno, controllo se è l'elemento el
            if (tab[pos].el.equals(el)) {
                if (tab[pos].isCancelled)
                    // l'elemento è già cancellato
                    return false;
                else {// cancello l'elemento marcandolo come cancellato
                    tab[pos].isCancelled = true;
                    this.numeroDiElementi--;
                    return true;
                }
            }
            // Non sono uguali, vado avanti nella catena di tentativi
            i++;
        } while (i < this.tab.length);
        // non ho trovato l'elemento nella catena di tentativi
        return false;
    }

    /*
     * Classe di comodo per gestire il flag "deleted" degli elementi eliminati
     * dalla tabella.
     */
    private class Element {
        private E el;

        private boolean isCancelled;

        Element(E el) {
            this.el = el;
            this.isCancelled = false;
        }

        @Override
        public int hashCode() {
            final int prime = 31;
            int result = 1;
            result = prime * result + getOuterType().hashCode();
            result = prime * result + ((el == null) ? 0 : el.hashCode());
            return result;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (!(obj instanceof OpenAddressingHashTable<?>.Element))
                return false;
            @SuppressWarnings("unchecked")
            Element other = (Element) obj;
            if (!getOuterType().equals(other.getOuterType()))
                return false;
            if (el == null) {
                if (other.el != null)
                    return false;
            } else if (!el.equals(other.el))
                return false;
            return true;
        }

        private OpenAddressingHashTable<E> getOuterType() {
            return OpenAddressingHashTable.this;
        }

    }

}
