/**
 * 
 */
package it.unicam.cs.asdl2021.es13sol;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * Classe che implementa un grafo non orientato tramite matrice di adiacenza.
 * Non sono accettate etichette dei nodi null e non sono accettate etichette
 * duplicate nei nodi (che in quel caso sono lo stesso nodo).
 * 
 * I nodi sono numerati da 0 a nodeCoount() - 1 seguendo l'ordine del loro
 * inserimento e la matrice di adiacenza ha dimensione nodeCount() *
 * nodeCount(). La posizione i,j della matrice è null se i nodi i e j non sono
 * collegati da un arco e contiene un oggetto della classe GraphEdge<L> se lo
 * sono. Tale oggetto rappresenta l'arco. Un oggetto uguale (secondo equals) e
 * con lo stesso peso (se gli archi sono pesati) deve essere presente nella
 * posizione j, i della matrice.
 * 
 * Questa classe non supporta i metodi di cancellazione di nodi e archi, ma
 * supporta tutti i metodi che usano indici, utilizzando l'indice assegnato a
 * ogni nodo in fase di inserimento.
 * 
 * @author Luca Tesei
 *
 */
public class AdjacencyMatrixUndirectedGraph<L> extends Graph<L> {
    /*
     * Le seguenti variabili istanza sono protected al solo scopo di agevolare
     * il JUnit testing
     */
    // per mantenere l'insieme dei nodi e ottimizzare la ricerca dell'indice di
    // un nodo creo una mappa hash
    protected HashMap<GraphNode<L>, Integer> nodesIndex;

    // matrice di adiacenza, gli elementi sono null o oggetti della classe
    // GraphEdge<L>
    protected ArrayList<ArrayList<GraphEdge<L>>> matrix;

    /**
     * Crea un grafo vuoto.
     */
    public AdjacencyMatrixUndirectedGraph() {
        this.matrix = new ArrayList<ArrayList<GraphEdge<L>>>();
        this.nodesIndex = new HashMap<GraphNode<L>, Integer>();
    }

    @Override
    public int nodeCount() {
        return this.nodesIndex.size();
    }

    @Override
    public int edgeCount() {
        // Scorro solo la parte triangolare superiore della matrice, in quanto
        // il grafo è non orientato e quindi gli archi "doppi" vanno contati una
        // sola volta. I cappi vengono anch'essi conteggiati.
        int count = 0;
        for (int i = 0; i < this.matrix.size(); i++)
            for (int j = i; j < this.matrix.get(i).size(); j++)
                if (this.matrix.get(i).get(j) != null)
                    count++;
        return count;
    }

    @Override
    public void clear() {
        this.matrix = new ArrayList<ArrayList<GraphEdge<L>>>();
        this.nodesIndex = new HashMap<GraphNode<L>, Integer>();
    }

    @Override
    public boolean isDirected() {
        // Questa classe implementa un grafo non orientato
        return false;
    }

    @Override
    public Set<GraphNode<L>> getNodes() {
        return this.nodesIndex.keySet();
    }

    @Override
    public boolean addNode(GraphNode<L> node) {
        if (node == null)
            throw new NullPointerException("Aggiunta di nodo null");
        if (containsNode(node))
            return false;
        // associo il nodo al nuovo indice nella mappa
        this.nodesIndex.put(node, this.nodeCount());
        // aggiungo una riga alla matrice
        this.matrix.add(new ArrayList<GraphEdge<L>>());
        // aggiungo null a tutte le colonne in corrispondenza della nuova riga
        for (int i = 0; i < this.nodeCount(); i++)
            this.matrix.get(this.nodeCount() - 1).add(null);
        // aggiungo una cella in fondo a ogni altra riga della matrice tranne
        // che per l'ultima riga
        for (int i = 0; i < this.nodeCount() - 1; i++)
            this.matrix.get(i).add(null);
        return true;
    }

    @Override
    public boolean removeNode(GraphNode<L> node) {
        throw new UnsupportedOperationException(
                "Remove di nodi non supportata");
    }

    @Override
    public boolean containsNode(GraphNode<L> node) {
        if (node == null)
            throw new NullPointerException("Ricerca di nodo null");
        return this.nodesIndex.containsKey(node);
    }

    @Override
    public GraphNode<L> getNodeOf(L label) {
        if (label == null)
            throw new NullPointerException(
                    "Ricerca di nodo con etichetta null");
        for (GraphNode<L> n : this.nodesIndex.keySet())
            if (n.getLabel().equals(label))
                return n;
        return null;
    }

    @Override
    public int getNodeIndexOf(L label) {
        if (label == null)
            throw new NullPointerException(
                    "Ricerca di nodo con etichetta null");
        GraphNode<L> n = getNodeOf(label);
        if (n == null)
            throw new IllegalArgumentException(
                    "Richiesta di indice di etichetta non esistente");
        else
            return this.nodesIndex.get(n);
    }

    @Override
    public GraphNode<L> getNodeAtIndex(int i) {
        if (i < 0 || i >= nodeCount())
            throw new IndexOutOfBoundsException(
                    "Richiesta di un nodo con indice inesistente");
        for (Map.Entry<GraphNode<L>, Integer> n : this.nodesIndex.entrySet())
            if (n.getValue().intValue() == i)
                return n.getKey();
        return null;
    }

    @Override
    public Set<GraphNode<L>> getAdjacentNodesOf(GraphNode<L> node) {
        if (node == null)
            throw new NullPointerException(
                    "Richiesta dei nodi adiacenti a un nodo null");
        // Cerco l'indice del nodo
        Integer i = this.nodesIndex.get(node);
        if (i == null)
            throw new IllegalArgumentException(
                    "Richiesta dei nodi adiacenti a un nodo non esistente");
        Set<GraphNode<L>> result = new HashSet<GraphNode<L>>();
        // Scorro la riga i della matrice e accumulo il risultato
        for (int j = 0; j < nodeCount(); j++)
            if (this.matrix.get(i.intValue()).get(j) != null) {
                GraphEdge<L> e = this.matrix.get(i.intValue()).get(j);
                if (e.getNode1().equals(node))
                    result.add(e.getNode2());
                else
                    result.add(e.getNode1());
            }
        return result;
    }

    @Override
    public Set<GraphNode<L>> getPredecessorNodesOf(GraphNode<L> node) {
        throw new UnsupportedOperationException(
                "Operazione non supportata in un grafo non orientato");
    }

    @Override
    public Set<GraphEdge<L>> getEdges() {
        Set<GraphEdge<L>> result = new HashSet<GraphEdge<L>>();
        // Scorro la parte triangolare superiore della matrice e accumulo il
        // risultato
        for (int i = 0; i < this.matrix.size(); i++)
            for (int j = i; j < this.matrix.get(i).size(); j++)
                if (this.matrix.get(i).get(j) != null)
                    result.add(this.matrix.get(i).get(j));
        return result;
    }

    @Override
    public boolean addEdge(GraphEdge<L> edge) {
        if (edge == null)
            throw new NullPointerException(
                    "Richiesta di inserire un arco null");
        Integer i = this.nodesIndex.get(edge.getNode1());
        if (i == null)
            throw new IllegalArgumentException(
                    "Richiesta di inserire un arco tra almeno un nodo non esistente");
        Integer j = this.nodesIndex.get(edge.getNode2());
        if (j == null)
            throw new IllegalArgumentException(
                    "Richiesta di inserire un arco tra almeno un nodo non esistente");
        if (edge.isDirected())
            throw new IllegalArgumentException(
                    "Richiesta di inserire un arco orientato in un grafo non orientato");
        // Inserisco l'arco nelle posizioni i,j e j,i se non già occupate
        if (this.matrix.get(i.intValue()).get(j.intValue()) != null)
            return false; // arco già presente
        this.matrix.get(i).set(j, edge);
        this.matrix.get(j).set(i, edge);
        return true;
    }

    @Override
    public boolean removeEdge(GraphEdge<L> edge) {
        throw new UnsupportedOperationException(
                "Operazione di remove non supportata in questa classe");
    }

    @Override
    public boolean containsEdge(GraphEdge<L> edge) {
        if (edge == null)
            throw new NullPointerException(
                    "Richiesta di ricerca di un arco null");
        Integer i = this.nodesIndex.get(edge.getNode1());
        if (i == null)
            throw new IllegalArgumentException(
                    "Richiesta di ricerca di un arco tra almeno un nodo non esistente");
        Integer j = this.nodesIndex.get(edge.getNode2());
        if (j == null)
            throw new IllegalArgumentException(
                    "Richiesta di ricerca di un arco tra almeno un nodo non esistente");
        return this.matrix.get(i).get(j) != null;
    }

    @Override
    public Set<GraphEdge<L>> getEdgesOf(GraphNode<L> node) {
        if (node == null)
            throw new NullPointerException(
                    "Richiesta degli archi connessi a un nodo null");
        // Cerco l'indice del nodo
        Integer i = this.nodesIndex.get(node);
        if (i == null)
            throw new IllegalArgumentException(
                    "Richiesta degli archi connessi a un nodo non esistente");
        Set<GraphEdge<L>> result = new HashSet<GraphEdge<L>>();
        // Scorro la riga i della matrice e accumulo il risultato
        for (int j = 0; j < nodeCount(); j++)
            if (this.matrix.get(i.intValue()).get(j) != null)
                result.add(this.matrix.get(i).get(j));
        return result;
    }

    @Override
    public Set<GraphEdge<L>> getIngoingEdgesOf(GraphNode<L> node) {
        throw new UnsupportedOperationException(
                "Operazione non supportata in un grafo non orientato");
    }

}
