package it.unicam.cs.asdl2021.es13sol;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * Gli oggetti di questa classe sono calcolatori di cammini minimi con sorgente
 * singola su un certo grafo orientato e pesato dato. Il grafo su cui lavorare
 * deve essere passato quando l'oggetto calcolatore viene costruito e non può
 * contenere archi con pesi negativi. Il calcolatore implementa il classico
 * algoritmo di Dijkstra per i cammini minimi con sorgente singola utilizzando
 * una coda con priorità che estrae il minimo in tempo lineare rispetto alla
 * lunghezza della coda. In questo caso il tempo di esecuzione dell'algoritmo di
 * Dijkstra è {@code O(n^2)} dove {@code n} è il numero dei nodi del grafo. Una
 * implementazione più efficiente richiederebbe una coda di priorità che
 * realizza le operazioni di estrazione del minimo e di decreasePriority in
 * tempo logaritmico.
 * 
 * @author Luca Tesei
 *
 * @param <L>
 *                il tipo delle etichette dei nodi del grafo
 */
public class SimpleDijkstraShortestPathComputer<L> {

    // TODO inserire le variabili istanza necessarie

    private final Graph<L> graph;

    private boolean computed;

    private GraphNode<L> lastSource;

    /**
     * Crea un calcolatore di cammini minimi a sorgente singola per un grafo
     * orientato e pesato privo di pesi negativi.
     * 
     * @param graph
     *                  il grafo su cui opera il calcolatore di cammini minimi
     * @throws NullPointerException
     *                                      se il grafo passato è nullo
     * 
     * @throws IllegalArgumentException
     *                                      se il grafo passato è vuoto
     * 
     * @throws IllegalArgumentException
     *                                      se il grafo passato non è orientato
     * 
     * @throws IllegalArgumentException
     *                                      se il grafo passato non è pesato,
     *                                      cioè esiste almeno un arco il cui
     *                                      peso è {@code Double.NaN}.
     * @throws IllegalArgumentException
     *                                      se il grafo passato contiene almeno
     *                                      un peso negativo.
     */
    public SimpleDijkstraShortestPathComputer(Graph<L> graph) {
        if (graph == null)
            throw new NullPointerException(
                    "Tentativo di costruire un calcolatore di cammini minimi su un grafo nullo");
        if (graph.isEmpty())
            throw new IllegalArgumentException(
                    "Tentativo di costruire un calcolatore di cammini minimi su un grafo vuoto");
        if (!graph.isDirected())
            throw new IllegalArgumentException(
                    "Tentativo di costruire un calcolatore di cammini minimi su un grafo non orientato");
        // Determina se tutti gli archi del grafo hanno un peso assegnato e, se
        // sì, positivo o nullo
        Set<GraphEdge<L>> edges = graph.getEdges();
        for (GraphEdge<L> e : edges) {
            if (!e.hasWeight())
                throw new IllegalArgumentException(
                        "Tentativo di costruire un calcolatore di cammini minimi su un grafo con almeno un arco con peso non specificato");
            if (e.getWeight() < 0)
                throw new IllegalArgumentException(
                        "Tentativo di costruire un calcolatore di cammini minimi su un grafo con almeno un arco con peso negativo");
        }
        // Il grafo è OK
        this.graph = graph;
        this.computed = false;
    }

    /**
     * Inizializza le informazioni necessarie associate ai nodi del grafo
     * associato a questo calcolatore ed esegue un algoritmo per il calcolo dei
     * cammini minimi a partire da una sorgente data.
     * 
     * @param sourceNode
     *                       il nodo sorgente da cui calcolare i cammini minimi
     *                       verso tutti gli altri nodi del grafo
     * @throws NullPointerException
     *                                      se il nodo passato è nullo
     * 
     * @throws IllegalArgumentException
     *                                      se il nodo passato non esiste nel
     *                                      grafo associato a questo calcolatore
     *
     */
    public void computeShortestPathsFrom(GraphNode<L> sourceNode) {
        if (sourceNode == null)
            throw new NullPointerException(
                    "Tentativo di calcolare i cammini minimi da un nodo sorgente nullo");
        // Determina i nodi del grafo
        Set<GraphNode<L>> nodes = this.graph.getNodes();
        if (!nodes.contains(sourceNode))
            throw new IllegalArgumentException(
                    "Tentativo di calcolare i cammini minimi da un nodo sorgente non esistente");
        // Creo la coda per i nodi utilizzando un semplice ArrayList
        ArrayList<GraphNode<L>> queue = new ArrayList<GraphNode<L>>();
        // Inizializza le distanze e i predecessori dei nodi del grafo e
        // inserisce i nodi nella coda
        for (GraphNode<L> n : nodes)
            if (n.equals(sourceNode)) {
                n.setFloatingPointDistance(0);
                n.setPrevious(null);
            } else {
                n.setFloatingPointDistance(Double.POSITIVE_INFINITY);
                n.setPrevious(null);
                queue.add(n);
            }
        // All'inizio il nodo con distanza minore è la sorgente, che non è stata
        // inserita nella coda
        GraphNode<L> currentNode = sourceNode;
        // Ciclo su tutti i nodi rimanenti
        do {
            // Determino gli archi uscenti dal nodo corrente
            Set<GraphEdge<L>> outgoingEdges = this.graph
                    .getEdgesOf(currentNode);
            // Rilasso le distanze di tutti gli archi collegati
            for (GraphEdge<L> e : outgoingEdges)
                relax(e);
            // Determino il prossimo nodo da trattare, se c'è
            if (queue.isEmpty())
                // ho finito
                break;
            else
                currentNode = extractMin(queue);
        } while (true);

        // Aggiorno la sorgente
        this.lastSource = sourceNode;

        // Aggiorno computed
        this.computed = true;

    }

    /*
     * Non utilizza una struttura dati per rendere logaritmica o costante questa
     * operazione. Il costo è lineare nella lunghezza della coda.
     */
    private GraphNode<L> extractMin(ArrayList<GraphNode<L>> queue) {
        if (queue.isEmpty())
            return null;
        // Cerco l'indice dell'elemento che ha peso floating point minimo
        int minimumIndex = 0;
        for (int i = 1; i < queue.size(); i++)
            if (queue.get(minimumIndex).getFloatingPointDistance() > queue
                    .get(i).getFloatingPointDistance())
                minimumIndex = i;
        // Elimino l'elemento minimo e lo restituisco
        GraphNode<L> result = queue.get(minimumIndex);
        queue.remove(minimumIndex);
        return result;
    }

    private void relax(GraphEdge<L> e) {
        // controllo se la distanza stimata attuale del nodo di arrivo è
        // maggiore di quella che si ottiene sommando la distanza stimata
        // attuale del nodo di partenza più il peso di questo arco
        if (e.getNode2().getFloatingPointDistance() > e.getNode1()
                .getFloatingPointDistance() + e.getWeight()) {
            // aggiorno la distanza stimata del nodo di arrivo
            e.getNode2().setFloatingPointDistance(
                    e.getNode1().getFloatingPointDistance() + e.getWeight());
            // metto come nodo predecessore del nodo di arrivo il nodo di
            // partenza
            e.getNode2().setPrevious(e.getNode1());
        }

    }

    /**
     * Determina se è stata invocata almeno una volta la procedura di calcolo
     * dei cammini minimi a partire da un certo nodo sorgente specificato.
     * 
     * @return true, se i cammini minimi da un certo nodo sorgente sono stati
     *         calcolati almeno una volta da questo calcolatore
     */
    public boolean isComputed() {
        return this.computed;
    }

    /**
     * Restituisce il nodo sorgente specificato nell'ultima chiamata effettuata
     * a {@code computeShortestPathsFrom(GraphNode<L>)}.
     * 
     * @return il nodo sorgente specificato nell'ultimo calcolo dei cammini
     *         minimi effettuato
     * 
     * @throws IllegalStateException
     *                                   se non è stato eseguito nemmeno una
     *                                   volta il calcolo dei cammini minimi a
     *                                   partire da un nodo sorgente
     */
    public GraphNode<L> getLastSource() {
        if (!this.computed)
            throw new IllegalStateException(
                    "Tentativo di ottenere il nodo sorgente senza che sia "
                            + "stato eseguito nemmeno una volta il calcolo dei cammini "
                            + "minimi");
        return this.lastSource;
    }

    /**
     * Restituisce il grafo su cui opera questo calcolatore.
     * 
     * @return il grafo su cui opera questo calcolatore
     */
    public Graph<L> getGraph() {
        return this.graph;
    }

    /**
     * Restituisce una lista di archi dal nodo sorgente dell'ultimo calcolo di
     * cammini minimi al nodo passato. Tale lista corrisponde a un cammino
     * minimo tra il nodo sorgente e il nodo target passato.
     * 
     * @param targetNode
     *                       il nodo verso cui restituire il cammino minimo
     *                       dalla sorgente
     * @return la lista di archi corrispondente al cammino minimo; la lista è
     *         vuota se il nodo passato è il nodo sorgente. Viene restituito
     *         {@code null} se il nodo passato non è raggiungibile dalla
     *         sorgente
     * 
     * @throws NullPointerException
     *                                      se il nodo passato è nullo
     * 
     * @throws IllegalArgumentException
     *                                      se il nodo passato non esiste
     * 
     * @throws IllegalStateException
     *                                      se non è stato eseguito nemmeno una
     *                                      volta il calcolo dei cammini minimi
     *                                      a partire da un nodo sorgente
     * 
     */
    public List<GraphEdge<L>> getShortestPathTo(GraphNode<L> targetNode) {
        if (!this.computed)
            throw new IllegalStateException(
                    "Tentativo di ottenere un cammino minimo senza che sia "
                            + "stato eseguito nemmeno una volta il calcolo dei cammini "
                            + "minimi");
        if (targetNode == null)
            throw new NullPointerException(
                    "Tentativo di ottenere un cammino minimo verso un nodo nullo");
        if (!this.graph.containsNode(targetNode))
            throw new IllegalArgumentException(
                    "Tentativo di ottenere un cammino minimo verso un nodo non esistente");
        ArrayList<GraphEdge<L>> result = new ArrayList<GraphEdge<L>>();
        try {
            // Chiamo la procedura di costruzione del path sul nodo reale del
            // grafo, quello passato potrebbe essere solo un nodo equals per
            // etichetta, ma con valori diversi degli attributi
            constructPath(this.graph.getNodeOf(targetNode.getLabel()), result);
        } catch (IllegalArgumentException e) {
            // Il nodo passato non è raggiungibile dalla sorgente
            return null;
        }
        return result;
    }

    private void constructPath(GraphNode<L> currentNode,
            ArrayList<GraphEdge<L>> path) {
        if (currentNode.equals(this.lastSource))
            return;
        if (currentNode.getPrevious() == null)
            throw new IllegalArgumentException(
                    "Tentativo di ottenere un cammino minimo verso un nodo non"
                            + " raggiungibile dalla sorgente");
        constructPath(currentNode.getPrevious(), path);
        // Determina l'arco dal nodo previous al nodo corrente e lo aggiunge al path
        for (GraphEdge<L> e : this.graph.getIngoingEdgesOf(currentNode)) {
            if (e.getNode1().equals(currentNode.getPrevious()))
                path.add(e);
        }
    }

    /**
     * Genera una stringa di descrizione di un path riportando i nodi
     * attraversati e i pesi degli archi. Nel caso di cammino vuoto genera solo
     * la stringa {@code "[ ]"}.
     * 
     * @param path
     *                 un cammino minimo
     * @return una stringa di descrizione del cammino minimo
     * @throws NullPointerException
     *                                  se il cammino passato è nullo
     */
    public String printPath(List<GraphEdge<L>> path) {
        if (path == null)
            throw new NullPointerException(
                    "Richiesta di stampare un path nullo");
        if (path.isEmpty())
            return "[ ]";
        // Costruisco la stringa
        StringBuffer s = new StringBuffer();
        s.append("[ " + path.get(0).getNode1().toString());
        for (int i = 0; i < path.size(); i++)
            s.append(" -- " + path.get(i).getWeight() + " --> "
                    + path.get(i).getNode2().toString());
        s.append(" ]");
        return s.toString();
    }

    // TODO inserire eventuali altri metodi accessori
}
