/**
 * 
 */
package it.unicam.cs.asdl2021.slides.ereditarieta;

/**
 * Un particolare conto di risparmio vincolato con un tasso aggiuntivo sui mesi
 * in cui l'importo è vincolato. Se non c'è nessuna operazione di prelievo nei
 * mesi vincolati alla fine del periodo del vincolo vengono accreditati gli
 * interessi aggiuntivi maturati. Se c'è una operazione di prelievo prima della
 * fine del vincolo si ricominciano a contare i mesi dall'inizio. Dopo la fine
 * del primo periodo di vincolo il contratto deve essere esplicitamente
 * rinnovato per continuare ad avere il tasso aggiuntivo.
 * 
 * @author Luca Tesei
 *
 */
public class TimedAccount extends SavingsAccount {
    private int numeroMesiVincolo;

    private int mesiRimanenti;

    private double tassoAggiuntivo;

    private boolean contrattoAttivo;

    /**
     * Costruisce un conto con contratto di vincolo attivo di default per 18
     * mesi al tasso del 3%.
     * 
     * @param saldo
     *                  saldo iniziale
     * @param nome
     *                  nome titolare
     * @param iban
     *                  codice IBAN
     */
    public TimedAccount(double saldo, String nome, String iban) {
        super(saldo, nome, iban);
        this.numeroMesiVincolo = 18;
        this.mesiRimanenti = 18;
        this.tassoAggiuntivo = 3.0;
        this.contrattoAttivo = true;
    }

    /**
     * Costruisce un conto con contratto di vincolo attivo di default per 18
     * mesi a un tasso specifico.
     * 
     * @param saldo
     *                             saldo iniziale
     * @param nome
     *                             nome titolare
     * @param iban
     *                             codice IBAN
     * @param tassoAggiuntivo
     *                             tasso di interesse
     */
    public TimedAccount(double saldo, String nome, String iban,
            double tassoAggiuntivo) {
        super(saldo, nome, iban, tassoAggiuntivo);
        this.numeroMesiVincolo = 18;
        this.mesiRimanenti = 18;
        this.tassoAggiuntivo = 3.0;
        this.contrattoAttivo = true;
    }

    /**
     * Costruisce un conto con contratto di vincolo attivo per un numero di mesi
     * specifico a un tasso specifico.
     * 
     * @param saldo
     *                              saldo iniziale
     * @param nome
     *                              nome titolare
     * @param iban
     *                              codice IBAN
     * @param numeroMesiVincolo
     *                              durata del vincolo in mesi
     * @param tassoAggiuntivo
     *                              tasso di interesse
     */
    public TimedAccount(double saldo, String nome, String iban,
            int numeroMesiVincolo, double tassoAggiuntivo) {
        super(saldo, nome, iban);
        this.numeroMesiVincolo = numeroMesiVincolo;
        this.tassoAggiuntivo = tassoAggiuntivo;
        this.mesiRimanenti = numeroMesiVincolo;
        this.contrattoAttivo = true;
    }

    /*
     * Un prelievo fa ripartire il numero di mesi del vincolo
     */
    public void withdraw(double importo) {
        super.withdraw(importo);
        if (this.contrattoAttivo) {
            if (this.mesiRimanenti > 0)
                // Rotto il vincolo, riparte il conteggio dei mesi
                this.mesiRimanenti = this.numeroMesiVincolo;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see it.unicam.cs.asdl1617.bankaccounts.SavingsAccount#endOfMonth()
     */
    @Override
    public void endOfMonth() {
        super.endOfMonth();
        if (this.mesiRimanenti > 1)
            this.mesiRimanenti--;
        else if (this.mesiRimanenti == 1) {
            // calcolo interesse aggiuntivo
            //
            double interesseMensile = (this.getSaldo() * this.tassoAggiuntivo
                    / 100) / 12;
            double interesse = interesseMensile * this.numeroMesiVincolo;
            super.deposit(interesse);
            this.mesiRimanenti = 0;
            this.contrattoAttivo = false;
        }
    }

    /**
     * Rinnova il contratto per un ulteriore periodo.
     * 
     * @param numeroMesiVincolo
     * @param tassoAggiuntivo
     * 
     * @throws IllegalStateException
     *                                   se viene chiamato durante un periodo di
     *                                   vincolo attivo.
     */
    public void renewContract(int numeroMesiVincolo, double tassoAggiuntivo) {
        if (this.contrattoAttivo)
            throw new IllegalStateException(
                    "Tentativo di rinnovare un contratto non ancora scaduto");
        this.numeroMesiVincolo = numeroMesiVincolo;
        this.mesiRimanenti = numeroMesiVincolo;
        this.tassoAggiuntivo = tassoAggiuntivo;
        this.contrattoAttivo = true;
    }

}
