package it.unicam.cs.asdl2021.slides.ereditarieta;

/**
 * Banca che gestisce la collezione di conti con un array.
 * 
 * @author Luca Tesei
 *
 */
public class Bank {

    private final String swiftCode;

    private String name;

    private String address;

    private BankAccount[] conti;

    private int lastFreePos;

    /**
     * @param swiftCode
     *                      codice identificativo della banca
     * @param name
     *                      nome della banca
     * @param address
     *                      indirizzo
     * @param dimension
     *                      numero di conti gestibili
     */
    public Bank(String swiftCode, String name, String address, int dimension) {
        this.swiftCode = swiftCode;
        this.name = name;
        this.address = address;
        this.conti = new BankAccount[dimension];
        this.lastFreePos = 0;
    }

    /**
     * @return the name
     */
    public String getName() {
        return name;
    }

    /**
     * @param name
     *                 the name to set
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * @return the address
     */
    public String getAddress() {
        return address;
    }

    /**
     * @param address
     *                    the address to set
     */
    public void setAddress(String address) {
        this.address = address;
    }

    /**
     * @return the swiftCode
     */
    public String getSwiftCode() {
        return swiftCode;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result
                + ((swiftCode == null) ? 0 : swiftCode.hashCode());
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (!(obj instanceof Bank))
            return false;
        Bank other = (Bank) obj;
        if (swiftCode == null) {
            if (other.swiftCode != null)
                return false;
        } else if (!swiftCode.equals(other.swiftCode))
            return false;
        return true;
    }

    /**
     * Inserisce un nuovo conto.
     * 
     * @param b
     *              il nuovo conto
     * @throws IllegalArgumentException
     *                                       se il conto è già esistente in
     *                                       questa banca
     * @throws IndexOutOfBoundsException
     *                                       se il numero massimo dei conti per
     *                                       questa banca è stato raggiunto
     */
    public void add(BankAccount b) {
        // Controlla che il conto non è già stato inserito
        if (search(b))
            throw new IllegalArgumentException(
                    "Inserimento di conto già esistente");
        // Inserisco il conto se c'è spazio sufficiente
        if (this.lastFreePos < this.conti.length) {
            this.conti[this.lastFreePos] = b;
            this.lastFreePos++;
        } else
            throw new IndexOutOfBoundsException(
                    "Tentativo di creare un conto in eccesso rispetto a quelli gestibili");
    }

    /**
     * Esegue le procedure di fine mese su tutti i conti.
     */
    public void endOfMonth() {
        // chiamo il metodo su tutti i conti
        for (int i = 0; i < this.lastFreePos; i++)
            this.conti[i].endOfMonth();
    }

    /**
     * Cerca se un conto è già presente
     * 
     * @param b
     *              un conto
     * @return true se il conto esiste già, false altrimenti
     */
    public boolean search(BankAccount b) {
        // Ricerco il conto nell'array
        int i = 0;
        boolean trovato = false;
        while (i < this.lastFreePos && !trovato)
            if (this.conti[i].equals(b)) // si basa su equals di BankAccount!
                trovato = true;
            else
                i++;
        return trovato;
    }

    /**
     * Cerca l'indice di un conto nella lista dei conti.
     * 
     * @param b
     *              un conto
     * @return la posizione del conto nella lista dei conti, -1 se il conto non
     *         esiste.
     */
    public int indexOf(BankAccount b) {
        // Ricerco il conto nell'array
        int i = 0;
        boolean trovato = false;
        while (i < this.lastFreePos && !trovato)
            if (this.conti[i].equals(b))
                trovato = true;
            else
                i++;
        if (trovato)
            return i;
        else
            return -1;
    }

    /**
     * Rimuove un certo conto.
     * 
     * @param b
     *              il conto da rimuovere
     * @return true se il conto è stato rimosso, false altrimenti
     */
    public boolean remove(BankAccount b) {
        // cerco la posizione del conto
        int p = this.indexOf(b);
        if (p == -1) // il conto non c'è, non rimuovo niente
            return false;
        // rimuovo il conto in posizione p
        // faccio lo shift a sinistra di tutti i conti a destra di p
        for (int i = p; i < this.lastFreePos - 1; i++)
            this.conti[i] = this.conti[i + 1];
        this.conti[this.lastFreePos - 1] = null;
        this.lastFreePos--;
        return true;
    }

}
