package it.unicam.cs.asdl2021.slides.ereditarieta;


/**
 * 
 * Conto corrente di base, senza nessuna particolare caratteristica.
 * 
 * @author Luca Tesei
 *
 */
public abstract class BankAccount implements Comparable<BankAccount> {

    
    private double saldo;

    private String nome;

    private final String iban;

    /**
     * @param saldo
     *                  saldo iniziale
     * @param nome
     *                  nome dell'intestatario
     * @param iban
     *                  codice unico identificativo del conto, immutabile
     */
    public BankAccount(double saldo, String nome, String iban) {
        this.saldo = saldo;
        this.nome = nome;
        this.iban = iban;
    }

    /**
     * @return the nome
     */
    public String getNome() {
        return nome;
    }

    /**
     * @param nome
     *                 the nome to set
     */
    public void setNome(String nome) {
        this.nome = nome;
    }

    /**
     * @return the saldo
     */
    public double getSaldo() {
        return saldo;
    }

    /**
     * @return the iban
     */
    public String getIban() {
        return iban;
    }


    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((iban == null) ? 0 : iban.hashCode());
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * l'equals di conti correnti si basa sull'iban, identificatore unico
     * immutabile
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (!(obj instanceof BankAccount))
            return false;
        BankAccount other = (BankAccount) obj;
        if (iban == null) {
            if (other.iban != null)
                return false;
        } else if (!iban.equals(other.iban))
            return false;
        return true;
    }

    /**
     * Metodo astratto che viene chiamato sul conto alla fine di ogni mese.
     * Esegue azioni specifiche del conto da fare alla fine del mese. Dipende
     * dalle varie sottoclassi.
     * 
     */
    public abstract void endOfMonth();

    /**
     * Deposita un importo nel conto.
     * 
     * @param importo
     *                    l'ammontare da depositare
     */
    public void deposit(double importo) {
        if (importo < 0)
            throw new IllegalArgumentException(
                    "Tentativo di deposito di importo negativo");
        saldo += importo;
    }

    /**
     * Preleva un importo dal conto.
     * 
     * @param importo
     *                    l'ammontare da prelevare
     */
    public void withdraw(double importo) {
        if (importo < 0)
            throw new IllegalArgumentException(
                    "Tentativo di prelievo di importo negativo");
        if (importo > saldo)
            throw new InsufficientFundException(
                    "Tentativo di prelevare più del posseduto");
        saldo -= importo;

    }

    /**
     * Ordinamento tra conti correnti tramite l'ordinamento delle stringhe iban.
     * 
     * @param other
     *                  il conto corrente da comparare con questo
     * @return un valore {@code < 0} se questo conto corrente precede other
     *         nell'ordinamento, 0 se i due conti correnti coincidono, un valore
     *         {@code > 0} se questo conto corrente segue other nell'ordinamento.
     */
    public int compareTo(BankAccount other) {
        return this.iban.compareTo(other.iban);
    }

}
