/*******************************************************************************
 * Copyright (c) 2019 Michele Loreti, Università di Camerino.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Contributors:
 *     Michele Loreti - initial API and implementation
 *******************************************************************************/
/**
 * 
 */
package it.unicam.cs.pa.battleship19;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.util.function.BiFunction;

/**
 * Una <i>view</i> basata su console.
 * 
 * @author loreti
 *
 */
public class ConsoleView implements BattleShipView {
	
	public static final char EMPTY_CHAR = ' ';
	public static final char SHIP_CHAR = '#';
	public static final char MISS_CHAR = '@';
	public static final char HIT_CHAR = '*';
	public static final char SUNK_CHAR = 'X';
	
	private PrintStream output;
	private BufferedReader input;
	private GameParameters parameters = GameParameters.DEFAULT_PARAMETERS;
	private FieldManager manager;
	private String id;
	private String spaceline;

	public ConsoleView( String id ) {
		this(id,System.out,new BufferedReader(new InputStreamReader(System.in)));
	}

	public ConsoleView(String id, PrintStream output, BufferedReader input) {
		this.id = id;
		this.input = input;
		this.output = output;
	}

	@Override
	public ShipLocation placeShip(int size) {
		try {
			while (true) {
				printField((i,j) -> (manager.isMyFieldEmpty(i,j)?EMPTY_CHAR:SHIP_CHAR));
				println("Posizionare nave di dimensione "+size+"...");
				Location l = inputLocation();
				Direction d = inputDirection(); 
				ShipLocation sl = new ShipLocation(l,d);
				if (manager.placeShip(size, sl)) {
					return sl;
				}
				println("ERRORE! La posizione no è valida!");
			}
		} catch (IOException e) {
			e.printStackTrace();
			return null;
		}
	}

	private void printField( BiFunction<Integer,Integer,Character> field ) {
		printHead();
		for( int i=0 ; i<parameters.getSize() ; i++ ) {
			String line = String.format("%3d ",(i+1));
			for( int j=0 ; j<parameters.getSize() ; j++ ) {
				line += String.format("| %c ", field.apply(i, j));
			}		
			line += "|";
			println(getSpaceLine());
			println(line);
		}
		println(getSpaceLine());
	}

	private String getSpaceLine() {
		if (spaceline == null) {
			spaceline = "    ";
			for( int j=0 ; j<parameters.getSize() ; j++ ) {
				spaceline += "+---";
			}		
			spaceline += "+";
		}
		return spaceline;
	}

	private void printHead() {
		String head = "    ";
		for( int i=0 ; i<parameters.getSize() ; i++ ) {
			head += String.format(" %3d", i+1);
		}
		println( head );
	}

	private void println(String string) {
		output.println(id+"> "+string);
	}


	private Direction inputDirection() throws IOException {
		while (true) {
			println("Inserire direzione della nave ('H' o 'V'):");
			String str = input.readLine();
			if (str.length()==1) {
				if (str.charAt(0)=='H') {
					return Direction.HORIZONTAL;
				}
				if (str.charAt(0)=='V') {
					return Direction.VERTICAL;
				}
			}
			System.out.println("Errore!");
		}		
	}

	private Location inputLocation() throws IOException {
		int row = inputInteger("Inserire riga: ",0,parameters.getSize());
		int column = inputInteger("Inserire colonna: ",0,parameters.getSize());
		return new Location(row,column);		
	}

	private int inputInteger(String message, int min, int max) throws IOException {
		while (true) {
			println(message);
			String str = input.readLine();
			try {
				int i = Integer.parseInt(str);
				if ((min<i)&&(i<=max)) {
					return i-1;
				}
			} catch (NumberFormatException e) {				
			}
			println("Inserire un valore intero tra "+(min+1)+" e "+max+" (compresi)!");
		}
	}

	@Override
	public Location launch() {
		try {
			printField(this::oppositeBattleFieldContent);
			println("Lanciare bomba!");
			Location l = inputLocation();
			println(String.format("Bomba lanciata a (%d,%d)!", l.getRow(),l.getRow()));
			return l;			
		} catch (IOException e) {
			e.printStackTrace();
			return null;
		}
	}
	
	private char myBattleFieldContent( int i , int j ) {
		LaunchResult lr = manager.getMyFieldStatus(i,j);
		if (lr == null) {
			return (manager.isMyFieldEmpty(i,j)?EMPTY_CHAR:SHIP_CHAR);
		} else {
			return charOf(lr.getType());
		}
	}

	private char oppositeBattleFieldContent( int i , int j ) {
		LaunchResult lr = manager.getOpponentFieldStatus(i, j);
		if (lr == null) {
			return ' ';
		} else {
			return charOf(lr.getType());
		}
	}

	private char charOf(ResultType type) {
		switch (type) {
		case MISS: 
			return MISS_CHAR;
		case HIT:
			return HIT_CHAR;
		case SUNK:
			return SUNK_CHAR;
		case FAIL:
			return ' ';
		}
		return ' ';
	}

	@Override
	public void setParameters(GameParameters parameters) {
		this.manager = new FieldManager(parameters.getSize());
	}

	@Override
	public void fight() {
		println("La battaglia inizia!");
	}

	@Override
	public void end(boolean winFlag, GameResult result) {
		if (winFlag) {
			println( "Hai vinto!");
		} else {
			println( "Hai perso... :-(");
		}
	}

	@Override
	public void bombAt(Location l) {
		println(String.format("Attacco ricevuto nella posizione (%d,%d)",l.getRow()+1,l.getColumn()+1));
		manager.bombAt(l);
	}


	@Override
	public void setLaunchResult(LaunchResult r) {
		println(launchMessage(r.getType()));
		manager.setLaunchResult(r);
		printField(this::myBattleFieldContent);
	}
	
	private String launchMessage(ResultType type) {
		switch (type) {
		case MISS: return "Acqua!";
		case FAIL: return "Errore! Nessun effetto...";
		case HIT: return "Colpito!";
		case SUNK: return "Affondato!";
		}
		return "";
	}

	public static void main(String[] argv) {
		ConsoleView cv = new ConsoleView("Prova");
		cv.setParameters(GameParameters.DEFAULT_PARAMETERS);
		System.out.println(cv.placeShip(2));
		System.out.println(cv.placeShip(2));
	}

}
