/*******************************************************************************
 * Copyright (c) 2019 Michele Loreti, Università di Camerino.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Contributors:
 *     Michele Loreti - initial API and implementation
 *******************************************************************************/
/**
 * 
 */
package it.unicam.cs.pa.battleship19;

import java.util.function.Function;

/**
 * <b>Responsabilità:</b> Coordina lo svolgimento di un match. 
 * 
 * @author loreti
 *
 */
public class MatchCoordinator {

	/**
	 * Array contenente i riferimenti ai due giocatori.
	 */
	private Player[] players;
	
	/**
	 * Array contenete i campi di battaglia dei due contendenti.
	 */
	private BattleField[] fields;
	
	/**
	 * Parametri usati nella partita.
	 */
	private GameParameters parameters;
	
	/**
	 * Turno attuale. 
	 */
	private int turn;

	/**
	 * Costruisce un nuovo match. Il costruttore prende come parametri:
	 * <ol>
	 * <li> <code>fielFactory</code>: una funzione utilizzata dal coordinatore per costruire i 
	 * campi di battaglia. Grazie all'uso di questo approccio, la classe <code>MatchCoordinator</code>
	 * &egrave; indipendente dalla struttura dati usata per memorizzare i field. 
	 * <li> <code>patameters</code>: i parametri del match.
	 * <li> <code>playerFactory1</code>: funzione usata per costruire il primo giocatore.
	 * <li> <code>playerFactory2</code>: funzione usata per costruire il secondo giocatore.
	 * </ol> 
	 * 
	 * La <i>responsabilit&agrave;</i> di costruire le componenti di una partita sono 
	 * lasciate al contesto dove il controllore viene utilizzato.
	 * 
	 * 
	 * @param fieldFactory funzione utilizzata per costruire i campi di battaglia usati dal controllore
	 * per memorizzare le informazioni della partita
	 * @param parameters parametri del gioco
	 * @param playerFactory1 funzione usata per costruire il primo giocatore
	 * @param playerFactory2 funzione usata per costruire il secondo giocatore
	 */
	public MatchCoordinator( 
			Function<GameParameters,BattleField> fieldFactory,
			GameParameters parameters,
			Function<GameParameters,Player> playerFactory1,
			Function<GameParameters,Player> playerFactory2
			) {
		this.parameters = parameters;
		this.players = new Player[] { 
				playerFactory1.apply(parameters), 
				playerFactory2.apply(parameters)
		};
		this.fields = new BattleField[] { fieldFactory.apply(parameters), fieldFactory.apply(parameters) };
		this.turn = 0;
	}
	
	/**
	 * Metodo invocato per iniziare la partita. La gestione del match si svolge in due fasi:
	 * <ol>
	 * <li> in una prima fase i giocatori piazzano le proprie navi nel campo di battaglia;
	 * <li> successivamente, fino a quando non c'è un vincitore, il controllore interroga i due 
	 * giocatori sulla posizione della bomba successiva.
	 * </ol>
	 * 
	 * Se durante l'esecuzione di una delle suddette attivit&agrave; viene sollevata un eccezione, 
	 * il giocatore perde in favore dell'altro. 
	 * 
	 * @return restituisce il risultato della partita. 
	 */
	public GameResult play( ) {
		try {
			placeShips(0);
			placeShips(1);	
			while (true) {
				manageTurn();
				if (fields[opponent()].getShips()==0) {
					return new Winner( current() );
				}
				turn = opponent();
			}
		} catch (BattleShipException e) {
			return new WinForFailure((e.getPlayerId()+1)%2,e);
		} 
	}

	/**
	 * Gestisce un round della partita. Questo consiste in tre attività:
	 * <ol>
	 * <li> ottiene dal giocatore di turno la posizione del lancio;
	 * <li> comunica al giocatore corrente il risultato del lancio;
	 * <li> comunica all'altro giocatore la posizione del lancio.
	 * </ol>
	 * 
	 * @throws IllegalLaunchException viene sollevata quando la locazione scelta
	 * per il lancio è illegale.
	 */
	private void manageTurn() throws IllegalLaunchException {
		Location l = players[current()].launch();
		players[current()].setResult( fields[opponent()].launch(l) );
		players[opponent()].bomb(l);
	}

	/**
	 * Restituisce l'indice del giocatore che attacca (turno).
	 * 
	 * @return l'indice del giocatore di turno.
	 */
	private int current() {
		return turn;
	}

	/**
	 * Restituisce l'indice del giocatore che riceve l'attacco.
	 * 
	 * @return l'indice del giocatore che riceve l'attacco.
	 */
	private int opponent() {
		return (turn+1)%2;
	}


	/**
	 * Si occupa di posizionare le navi del giocatore con indice <code>playerId</code>.
	 * 
	 * @param playerId indice del giocatore che piazza le navi.
	 * 
	 * @throws IllegalShipPlacement eccezione sollevata in caso di un errore nel piazzamento.
	 */
	private void placeShips(int playerId) throws IllegalShipPlacement {
		int[] ships = parameters.getFleet();
		for( int i=0 ; i<ships.length ; i++ ) {
			for( int j=0 ; j<ships[i] ; j++ ) {
				ShipLocation location = players[playerId].placeShip( i+1 );
				if (!fields[playerId].place( i+1 , location )) {
					throw new IllegalShipPlacement(playerId, "Illegal ship position ["+location+"]!");
				}
			}
		}
	}
	
}
