/*******************************************************************************
 * Copyright (c) 2019 Michele Loreti, Università di Camerino.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Contributors:
 *     Michele Loreti - initial API and implementation
 *******************************************************************************/
/**
 * 
 */
package it.unicam.cs.pa.battleship19;

/**
 * Rappresenta un campo di battaglia memorizzato attraverso una matrice.
 * 
 * @author loreti
 *
 */
public class MatrixBattleField implements BattleField {
	
	private final Ship[][] field;
	private final boolean[][] bombs;
	private int size; 
	private int ships;

	public MatrixBattleField( int size ) {
		this.size = size;
		this.field = new Ship[size][size];
		this.bombs = new boolean[size][size];
		this.ships = 0;
	}
	

	@Override
	public boolean isValid(Location l) {
		return (l.getRow()<size)&&(l.getColumn()<size);
	}

	@Override
	public LaunchResult launch(Location l) throws IllegalLaunchException {
		try {
			if (bombs[l.getRow()][l.getColumn()]) {
				return new LaunchResult(ResultType.FAIL,l);
			}
			bombs[l.getRow()][l.getColumn()] = true;
			Ship s = get(l);
			if (s == null) {
				return new LaunchResult(ResultType.MISS,l);
			}
			if (s.bomb()) {
				ships--;
				return new LaunchResult(ResultType.SUNK,s.locations());
			} else {
				return new LaunchResult(ResultType.HIT,l);
			}			
		} catch (IndexOutOfBoundsException e) {
			throw new IllegalLaunchException(0, "Illegal location");
		}
	}

	@Override
	public int getShips() {
		return ships;
	}


	@Override
	public boolean place(int size, ShipLocation shipLocation) {
		Ship ship = new Ship( shipLocation , size );
		Location[] locs = shipLocation.getLocations(size);
		if (check(locs)) {
			for (Location l : locs) {
				set(l,ship);
			}
			ships++;
			return true;
		} else {
			return false;
		}
	}


	private void set(Location l, Ship ship) {
		this.field[l.getRow()][l.getColumn()] = ship;
	}


	private boolean check(Location[] locations ) {
		for (Location l : locations) {
			if (!isValid(l)) {
				return false;
			}
			if (get(l) != null) {
				return false;
			}
		}
		return true;
	}


	private Ship get(Location l) {
		return this.field[l.getRow()][l.getColumn()];
	}


	public boolean isEmpty(int i, int j) {
		return this.field[i][j]==null;
	}


	public int size() {
		return this.field.length;
	}
	
}
