/*
 * Per approfondimenti sulle lambda expressions, link al tutorial Oracle:
 * https://docs.oracle.com/javase/tutorial/java/javaOO/lambdaexpressions.html
 */
import java.time.LocalDate;
import java.time.Month;
import java.util.ArrayList;
import java.util.List;

class Main
{
    public static void main(String[] args)
    {
        Main m = new Main();
        List<Persona> persone = new ArrayList<Persona>();
        persone.add(new Persona("Nome1", "Cognome1", LocalDate.of(1987, Month.NOVEMBER, 12)));
        persone.add(new Persona("Nome2", "Cognome2", LocalDate.of(2000, 5, 10)));
        persone.add(new Persona("Nome3", "Cognome3", LocalDate.of(1995, 3, 21)));
        persone.add(new Persona("Nome4", "Cognome4", LocalDate.of(1980, 11, 30)));
        persone.add(new Persona("Nome5", "Cognome5", LocalDate.of(2010, 7, 8)));
        
        //passo1: uso di un metodo con un parametro
        System.out.println("\nStampa dei minorenni usando un metodo con 1 parametro: ");
        m.stampaLista(m.getPersoneConEtaMinoreDi(persone, 18));

        //passo2: uso di un metodo con 2 parametri
        System.out.println("\nStampa dei minorenni usando un metodo con 2 parametri: ");
        m.stampaLista(m.getPersoneConEtaCompresaTra(persone, 0, 17));
        
        //passo3: uso di classe interna
        System.out.println("\nStampa dei minorenni usando una classe interna: ");
        m.stampaLista(m.filtraPersone(persone, m.new ControllaMinorenni()));
        
        //passo4: uso di classe anonima
        System.out.println("\nStampa dei minorenni usando una classe anonima: ");
        m.stampaLista(m.filtraPersone(persone, m.new ControllaMinorenni() {
            public boolean filtra(Persona p) {
                return p.getAnni() < 18;
            }
        }));
        
        //passo5: uso di lambda expression
        System.out.println("\nStampa dei minorenni usando una lambda expression: ");
        m.stampaLista(m.filtraPersone(persone, (Persona p) -> p.getAnni() >= 0 && p.getAnni() < 18));
        /*
         * Le lambda expressions rappresentano dei metodi anonimi
         * Sono utilizzate per implementare il metodo astratto di una interfaccia funzionale in maniera piu' coincisa
         * Sono formate da una serie di parametri contenuti tra parentesi, e una o piu' istruzioni, nella forma:
         * (parametri) -> istruzione
         * (parametri) -> { 
         *     istruzione1;
         *     istruzione2;
         *  }
         * Nel caso di istruzione singola, il compilatore la considera automaticamente come valore di ritorno del return
         * 
         * esempi:
         * () -> 5                                  //non prende in input alcun paramentro, e restituisce una costante
         * x -> 2 * x                               //prende in input un paramentro x (il tipo e' specificato nel metodo astratto) e restituisce il suo valore * 2
         * (x, y) -> x – y                          //prende in input due paramentri x e y, e restituisce il valore x - y
         * (String s) -> { System.out.print(s); }   //prende in input un parametro stringa s, e la stampa a video
         * (int x, int y) -> {                      //prende in input due parametri interi x e y, stampa e restituisce il valore x + y 
         *     int n = x + y; 
         *     System.out.print(n); 
         *     return n;
         *  }                 
         */
    }
    
    
    //passo1: uso di un metodo con due input
    public List<Persona> getPersoneConEtaMinoreDi(List<Persona> persone, int minEta) 
    {
        List<Persona> lista = new ArrayList<Persona>();
        for (Persona p : persone) {
            if (p.getAnni() < minEta) {
                lista.add(p);
            }
        }
        return lista;
    }
    
    //passo2: uso di un metodo piu' generico con tre input
    public List<Persona> getPersoneConEtaCompresaTra(List<Persona> persone, int minEta, int maxEta) 
    {
        List<Persona> lista = new ArrayList<Persona>();
        for (Persona p : persone) {
            int eta = p.getAnni();
            if (eta >= minEta && eta <= maxEta) {
                lista.add(p);
            }
        }
        return lista;
    }
    
    /*
     * passo3: uso di un metodo generico con due input, che richiede un'interfaccia come input
     * il filtro sara' una classe che implementa quell'interfaccia, fornendo il codice al suo metodo astratto
     * il metodo non applica esplicitamente alcun filtro, ma richiama il metodo dell'interfaccia
     */
    public List<Persona> filtraPersone(List<Persona> persone, FiltroPersona filtro)
    {
        List<Persona> lista = new ArrayList<Persona>();
        for(Persona p : persone) {
            //richiama il metodo filtra(Persona) dell'interfaccia
            //l'implementazione del filtro e' delegata alla classe che implementa l'interfaccia
            if (filtro.filtra(p))
                lista.add(p);
        }
        return lista;
    }
    
    
    //inner class
    class ControllaMinorenni implements FiltroPersona {
        //filtra le persone minorenni. Restituisce true se l'eta persona e' compresa tra 0 e 18 
        public boolean filtra(Persona p) {
            return p.getAnni() >= 0 && p.getAnni() < 18;
        }
    }
    
    
    //metodo per stampare una lista su console
    public void stampaLista(List<Persona> lista)
    {
        for(Persona p : lista)
            System.out.println(p.toString());
    }
}